/* --------------------------------------------------------------------------
 *
 * Copyright (C) 2007 Leif Erik Larsen, Kjerringvik, Norway.
 *
 * This file is part of the Open Source Edition of Larsen Commander, as
 * available from http://home.online.no/~leifel/lcmd/.  This code is free 
 * software; you can redistribute it and/or modify it under the terms of 
 * the GNU General Public License version 3 only, as published by the 
 * Free Software Foundation.  
 *
 * This code is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 * version 3 at http://www.gnu.org/licenses/gpl-3.0.txt for more details 
 * (a copy is included in the LICENSE file that accompanied this code).
 *
 * ------------------------------------------------------------------------ */

#ifndef __GLIB_STANDARDFILESYSTEM
#define __GLIB_STANDARDFILESYSTEM

#include "glib/vfs/GVfs.h"
#include "glib/util/GDriveInfo.h"
#include "glib/util/GKeyBag.h"
#include "glib/util/GHashtable.h"
#include "glib/primitives/GInteger.h"
#include "glib/primitives/GAtomicCounter.h"

/**
 * This class represents the default local file system, which can be of any 
 * type (physical disks, network drives, memory cards, etc.) as long as the 
 * operating system sees them as standard file system paths.
 *
 * @author  Leif Erik Larsen
 * @since   2002.07.28
 */
class GVfsLocal : public GVfs
{
   private:

      /** Current directory, including drive. TODO: But no trailing slash? */
      GString curDir;

      /** Cached information about the drive of the current directory. */
      mutable GDriveInfo driveInfo;

      /** The time stamp as of when {@link #driveInfo} was last updated. */
      mutable ulonglong lastDriveInfoUpdateTimeMillis;

      /**
       * @author  Leif Erik Larsen
       * @since   2005.03.29
       */
      class OpenFile : public GObject
      {
         public:
    
            /** The path used to open the file. */
            GString path;

            /** The system dependent handle of the open file. */
            GVfs::FileHandle sysFileHandle;

         public:
    
            OpenFile ( const GString& path, GVfs::FileHandle sysFileHandle );
            virtual ~OpenFile ();
      };

      static GHashtable<GInteger, GVfsLocal::OpenFile> OpenFiles;

      /**
       * Used by {@link #createTemporaryFile} to make a unique filename 
       * in a thread safe manner.
       *
       * @author  Leif Erik Larsen
       * @since   2005.09.04
       */
      static GAtomicCounter<int> TempFileCounter;

   public:

      /**
       * @author  Leif Erik Larsen
       * @since   2006.02.02
       * @see     #getSlashStr
       */
      static const GString SlashStr;

   public:

      /**
       * Creates a new standard file system object that initially 
       * has no "current drive and directory".
       */
      GVfsLocal ();

      virtual ~GVfsLocal ();

   private:

      /**
       * @author  Leif Erik Larsen
       * @since   2005.03.29
       */
      GVfsLocal::OpenFile* getOpenFile ( GVfs::FileHandle hfile );

   public:

      /**
       * Calculate the allocated file size, with respect to the 
       * specified "bytes per sector" count.
       *
       * @author  Leif Erik Larsen
       * @since   2005.03.29
       * @see     GDriveInfo#GetBytesPerSector
       */
      static ulonglong CalcFileSizeAlloc ( ulonglong fsize, int bytesPerSector );

      /**
       * Close the specified file, as recently opened by {@link openFile}.
       * See the declaring {@link GVfs#closeFile} for details.
       *
       * @author  Leif Erik Larsen
       * @since   2005.03.09
       */
      virtual GError closeFile ( FileHandle hfile );

      /**
       * Close the deletion handle recently returned by {@link #openDeletion}.
       * See the declaring {@link GVfs#closeDeletion} for details.
       *
       * @author  Leif Erik Larsen
       * @since   2005.08.31
       */
      virtual GError closeDeletion ( DeletionHandle hdel );

      /** 
       * Return true if the path contains at least one slash character. 
       *
       * @author  Leif Erik Larsen
       * @since   2005.09.01
       */
      virtual bool containsAnySlash ( const GString& path ) const;

      /**
       * Create the specified directory on the file system.
       * See the declaring {@link GVfs#createDirectory} for details.
       *
       * @author  Leif Erik Larsen
       * @since   2005.09.04
       */
      virtual GError createDirectory ( const GString& dir );

      /**
       * Create a file with a unique name in the specified directory, for 
       * temporary usage by the application. The file created will not 
       * be automatically deleted when the application exit.
       *
       * @author  Leif Erik Larsen
       * @since   2005.09.04
       * @param   prefix  Optional prefix string for the filename.
       *                  This is typically a two or three character 
       *                  abbreviation for the name of the calling 
       *                  application. Only the first four letters 
       *                  of the specified prefix string will be used.
       * @param   dir     The directory of where to create the temporary 
       *                  file, or an empty string to create it in the 
       *                  system default TEMP/TMP directory.
       * @throws  GIOException in case of any error.
       */
      GString createTemporaryFile ( const GString& prefix = GString::Empty, 
                                    const GString& dir = GString::Empty );

      /**
       * Load the filenames that match the specified filename search pattern 
       * and attributes, from the file system.
       *
       * @author  Leif Erik Larsen
       * @since   2004.12.23
       */
      virtual void fillList ( GVfs::List& list,
                              const GString& pattern,
                              bool inclFiles,
                              bool inclDirs,
                              bool inclHidden,
                              bool inclSys ) const;

      virtual int findFirst ( GFileItem& fitem, const GString& path = GString::Empty, bool* cancelSem = null, int* preLoadCounter = null ) const;
      virtual bool findFirstMightNeedLongTime () const;
      virtual bool findNext ( int hdir, GFileItem& fitem ) const;
      virtual void findClose ( int hdir ) const;

      /**
       * Current directory with drive, but without trailing slash except
       * if it is the root directory that is current. Sample return 
       * values: "C:\" or "C:\MyDir\SubDir".
       *
       * @author  Leif Erik Larsen
       * @since   2002.07.30
       * @param   slash   True if the returned directory string should 
       *                  always contain a trailing slash, even if it is 
       *                  not the root directory.
       */
      virtual GString getCurrentDirectory ( bool slash ) const;

      /**
       * Get a reference to the drive information of the current FS drive.
       *
       * @author  Leif Erik Larsen
       * @since   2004.05.04
       */
      const GDriveInfo& getCurrentDriveInfo () const;

      /**
       * Get the letter of the current drive, e.g. "A".
       *
       * @author  Leif Erik Larsen
       * @since   2002.07.28
       */
      char getCurrentDriveLetter () const;

      /**
       * Get the name (not the label) of the current drive, e.g. "A:".
       *
       * @author  Leif Erik Larsen
       * @since   2002.07.28
       */
      GString getCurrentDriveName () const;

      /**
       * Get the attributes (readonly, hidden, etc.) of the specified file.
       * See the declaring {@link GVfs#getFileAttributes} for details.
       *
       * @author  Leif Erik Larsen
       * @since   2005.03.11
       */
      virtual int getFileAttributes ( const GString& path, GError* errorCode = null );

      /**
       * Get the current seek position of the specified file (not pipe).
       * See the declaring {@link GVfs#getFileSeekPos} for details.
       *
       * @author  Leif Erik Larsen
       * @since   2005.03.11
       */
      virtual longlong getFileSeekPos ( GVfs::FileHandle hfile, GError* err );

      /**
       * Get the size of the specified open file.
       * See the declaring {@link GVfs#getFileSize} for details.
       *
       * @author  Leif Erik Larsen
       * @since   2005.03.14
       */
      virtual longlong getFileSize ( GVfs::FileHandle hfile, GError* err );

      /**
       * Get the short name of the implementing file system.
       * Typical return values are e.g.: "FAT", "FAT32", "NTFS", "HPFS", etc.
       *
       * @author  Leif Erik Larsen
       * @since   2004.12.23
       */
      virtual const GString& getFileSystemName () const;

      /**
       * @author  Leif Erik Larsen
       * @since   2007.02.05
       */
      virtual longlong getFreeDriveSpace () const;

      /**
       * Always return an empty string.
       *
       * @author  Leif Erik Larsen
       * @since   2002.08.20
       */
      virtual GString getLogicalSelfName () const;

      /**
       * Always return an empty string.
       *
       * @author  Leif Erik Larsen
       * @since   2002.08.20
       */
      virtual GString getPhysicalSelfName () const;

      /**
       * Get the root directory specification string (e.g. "A:\")
       * of the current directory of this file system instance.
       *
       * @author  Leif Erik Larsen
       * @since   2002.07.28
       */
      virtual GString getRootDir () const;

      /**
       * Get the string (usually a one character string) used to 
       * separate directory elements on this local file system.
       * On Windows and OS/2 System FS this is a backslash. 
       * On Linux System FS this is a forward slash.
       *
       * @author  Leif Erik Larsen
       * @since  2006.02.02
       */
      virtual const GString& getSlashStr () const;

      /**
       * Return true if and only if it is the root directory of the 
       * file system that is the current directory. For this class,
       * this is true if the current directory represents a drive 
       * only (e.g. "a:\" or "A:\").
       *
       * @author  Leif Erik Larsen
       * @since   2004.05.04
       */
      virtual bool isCurrentDirectoryTheRoot () const;

      /**
       * Return true if and only if the current directory is represented
       * on a classic FAT16 formatted drive.
       *
       * @author  Leif Erik Larsen
       * @since   2004.05.04
       */
      bool isFat16 () const;

      /**
       * Will always return false.
       *
       * @author  Leif Erik Larsen
       * @since   2004.12.23
       */
      virtual bool isFilePreparationPossiblyLengthy () const;

      /**
       * Same as {@link GFile::IsSlash}.
       *
       * @author  Leif Erik Larsen
       * @since   2004.12.23
       */
      virtual bool isSlash ( char chr ) const;

      /**
       * Load information about the specified file into the specified object.
       * See the declaring {@link GVfs#loadFileInfo} for details.
       *
       * @author  Leif Erik Larsen
       * @since   2005.03.29
       */
      virtual GError loadFileInfo ( GVfs::FileHandle hfile, GFileItem& info );

      /**
       * Implements {@link GVfs#moveOrRenameFile}.
       *
       * @author  Leif Erik Larsen
       * @since   2005.08.26
       */
      virtual GError moveOrRenameFile ( const GString& existingName, 
                                        const GString& newName,
                                        bool allowCopy );

      /**
       * Open a new deletion handle to be used with 
       * {@link #removeFile} and/or {@link #removeDirectory}.
       * See the declaring {@link GVfs#openDeletion} for details.
       *
       * @author  Leif Erik Larsen
       * @since   2005.08.31
       * @see     #closeDeletion
       */
      virtual DeletionHandle openDeletion ();

      /**
       * Open the specified file in the VFS.
       * See the declaring {@link GVfs#openFile} for details.
       *
       * @author  Leif Erik Larsen
       * @since   2005.03.09
       */
      virtual FileHandle openFile ( const GString& path, 
                                    GError* errorCode = null,
                                    OF_Mode modeOpt = Mode_ReadOnly,
                                    OF_Create createOpt = Create_Never,
                                    OF_Share shareOpt = Share_DenyWrite,
                                    int flagsOpt = OF_FLAG_SEQUENTIAL_ACCESS,
                                    OF_ActionTaken* actionTaken = null );

      /**
       * Perform all file and/or directory deletion operations requested
       * on the specified deletion handle.
       * See the declaring {@link GVfs#performDeletion} for details.
       *
       * @author  Leif Erik Larsen
       * @since   2005.08.31
       */
      virtual GError performDeletion ( DeletionHandle hdel );

      /**
       * @author  Leif Erik Larsen
       * @since   2004.12.23
       */
      virtual File* preparePhysicalFile ( const GString& fileName, 
                                          WorkStatus& stat,
                                          const GString& prefix );

      /**
       * Reads data from the specified file handle, which must have been 
       * opened for reading by {@link #openFile}, and positioned if needed.
       * See the declaring {@link GVfs#readFromFile} for details.
       *
       * @author  Leif Erik Larsen
       * @since   2005.03.09
       */
      virtual GError readFromFile ( GVfs::FileHandle hfile, 
                                    void* buff, 
                                    int numBytesToRead, 
                                    int* numBytesActuallyRead = null );

      /**
       * See the declaring {@link GVfs#removeDirectory} for details.
       *
       * @author  Leif Erik Larsen
       * @since   2005.03.22
       * @param   hdel A deletion handle opened with {@link #openDeletion}.
       */
      virtual GError removeDirectory ( DeletionHandle hdel, 
                                       const GString& path, 
                                       bool trashCan,
                                       HWND ownerWin = null );

      /**
       * See the declaring {@link GVfs#removeFile} for details.
       *
       * @author  Leif Erik Larsen
       * @since   2005.03.22
       * @param   hdel A deletion handle opened with {@link #openDeletion}.
       */
      virtual GError removeFile ( DeletionHandle hdel, 
                                  const GString& path, 
                                  bool trashCan );

      /**
       * Set the current directory with drive, but without trailing slash 
       * except if it is the root directory that is set. Sample of valid 
       * argument values: "C:\" or "C:\MyDir\SubDir".
       *
       * If the specified directory does not contain any drive specification
       * (e.g. "\MyDir\SubDir") then we will attempt setting the directory 
       * of the current drive. If the directory does not even contain a 
       * prefix slash ("e.g. "MyDir\SubDir") will still assume that the 
       * directory is starting at the root (just as if a prefix slash was 
       * specified).
       *
       * @author  Leif Erik Larsen
       * @since   2002.08.04
       */
      virtual GError setCurrentDirectory ( const GString& dir );

      /**
       * Set the attributes (readonly, hidden, etc.) of the specified file.
       * See the declaring {@link GVfs#setFileAttributes} for details.
       *
       * @author  Leif Erik Larsen
       * @since   2005.03.11
       */
      virtual GError setFileAttributes ( const GString& path, int attr );

      /**
       * Set the seek position of the specified file, relative to the
       * current position of the file.
       * See the declaring {@link GVfs#setFileSeekPosFromCurrent} for details.
       *
       * @author  Leif Erik Larsen
       * @since   2005.03.11
       */
      virtual GError setFileSeekPosFromCurrent ( GVfs::FileHandle hfile, 
                                                 longlong distanceToMove );

      /**
       * Set the seek position of the specified file, relative to the
       * end position of the file.
       * See the declaring {@link GVfs#setFileSeekPosFromEnd} for details.
       *
       * @author  Leif Erik Larsen
       * @since   2003.07.27
       */
      virtual GError setFileSeekPosFromEnd ( GVfs::FileHandle hfile, 
                                             longlong distanceToMove );

      /**
       * Set the seek position of the specified file, relative to the
       * start position of the file.
       * See the declaring {@link GVfs#setFileSeekPosFromStart} for details.
       *
       * @author  Leif Erik Larsen
       * @since   2003.07.27
       */
      virtual GError setFileSeekPosFromStart ( GVfs::FileHandle hfile, 
                                               longlong distanceToMove );

      /**
       * Set the size of the specified open file.
       * The file must have been opened for random write access.
       * See the declaring {@link GVfs#setFileSize} for details.
       *
       * @author  Leif Erik Larsen
       * @since   2005.03.11
       */
      virtual GError setFileSize ( GVfs::FileHandle hfile, longlong size );

      /**
       * Make sure the specified dir has a trailing slash, if not 
       * empty. The slash character is a backslash on Windows and 
       * OS/2, or a foreward slash on Linux.
       *
       * @author  Leif Erik Larsen
       * @since   2004.12.18
       */
      virtual GString& slash ( GString& dir ) const;

      /**
       * Convert foreward slashes (if any) to backslashes.
       * See the declaring {@link GVfs#translateSlashes} for details.
       *
       * @author  Leif Erik Larsen
       * @since   2005.03.11
       */
      virtual void translateSlashes ( GString& path ) const;

      /**
       * Update our internal information about the drive as of the current 
       * directory. If the parameter <i>force</i> if false then we will 
       * actually update only if the current directory has in fact changed,
       * else we will update anyway.
       *
       * @author  Leif Erik Larsen
       * @since   2004.05.04
       */
      void updateInternalDriveInfo ( bool force = false ) const;

      /**
       * Write attributes and time stamps of a file or directory.
       * See the declaring {@link GVfs#writeAttrAndTimes} for details.
       *
       * @author  Leif Erik Larsen
       * @since   2005.03.29
       */
      virtual GError writeAttrAndTimes ( GVfs::FileHandle hfile, 
                                         const class GFileItem& info,
                                         const GString& path = GString::Empty );

      /**
       * Writes data to the specified file handle, which must have been 
       * opened for writing by {@link #openFile}, and positioned if needed.
       * See the declaring {@link GVfs#writeToFile} for details.
       *
       * @author  Leif Erik Larsen
       * @since   2005.03.10
       */
      virtual GError writeToFile ( GVfs::FileHandle hfile, 
                                   const void* buff, 
                                   int numBytesToWrite, 
                                   int* numBytesActuallyWritten = null );

      virtual bool supportsChangeFileNameCase () const;
      virtual bool isFileNameCasePreserved () const;
      virtual bool isFileNameCaseSensitive () const;
};

#endif
