/* --------------------------------------------------------------------------
 *
 * Copyright (C) 2007 Leif Erik Larsen, Kjerringvik, Norway.
 *
 * This file is part of the Open Source Edition of Larsen Commander, as
 * available from http://home.online.no/~leifel/lcmd/.  This code is free 
 * software; you can redistribute it and/or modify it under the terms of 
 * the GNU General Public License version 3 only, as published by the 
 * Free Software Foundation.  
 *
 * This code is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 * version 3 at http://www.gnu.org/licenses/gpl-3.0.txt for more details 
 * (a copy is included in the LICENSE file that accompanied this code).
 *
 * ------------------------------------------------------------------------ */

#include "lcmd/LCmdConsoleMonitor.h"
#include "lcmd/LCmdCmdContainer.h"
#include "lcmd/LCmdDlgGotoLine.h"
#include "lcmd/LCmd.h"

#include "glib/gui/event/GKeyMessage.h"
#include "glib/gui/event/GUserMessage.h"

LCmdConsoleMonitor::LCmdConsoleMonitor ( LCmdCmdContainer& parentWin, 
                                         const GString& constraints )
                   :GTextViewer(&lcmd->conMonTextSearch, 
                                "Monitor", 
                                constraints, 
                                parentWin, 
                                true, 
                                true),
                    lineBuff(512),
                    lineBuffPos(0),
                    conMon(parentWin)
{
   defaultBackgroundColor = GColor::BLACK;
   defaultForegroundColor = GColor::YELLOW;
   defaultFontNameSize = lcmd->options.DefaultFontConMon;
   setBackgroundColor(defaultBackgroundColor);
   setForegroundColor(defaultForegroundColor);
   setFontNameSize(defaultFontNameSize);
   setAccelTable("ConsoleMonitorAccel");

   // Activate the popup menu of which to display when the user
   // right-click on the console monitor (text viewer) window.
   setPopupMenu("ConsoleMonitorMenu", lcmd->isUseFancyPopupMenues());
}

LCmdConsoleMonitor::~LCmdConsoleMonitor ()
{
}

void LCmdConsoleMonitor::onCaretPositionChanged ()
{
   lcmd->mainWin.updateStatusbarConsoleCurrentLine();
}

bool LCmdConsoleMonitor::onUserMessage ( GUserMessage& msg )
{
   GString id = msg.getParam1String();
   if (id != "AppendText")
      return GTextViewer::onUserMessage(msg);
   GObject* p2 = msg.getParam2();
   const GString& txt = dynamic_cast<GString&>(*p2);
   appendTextFromGuiThread(txt);
   return true;
}

bool LCmdConsoleMonitor::onKeyDown ( const GKeyMessage& key )
{
   if (key.isPureCharacter())
   {
      // Use this event to help the user put the caret back to the command 
      // line. This is probably what he wants since he has typed a character 
      // key on the keyboard.
      if (containsFocusedWindow())
      {
         conMon.cmdline.entry.grabFocus();
         char chr = key.getCharacter();
         if (chr >= 32)
         {
            GString keyStr = GCharacter::ToString(chr);
            conMon.cmdline.entry.replaceSelection(keyStr);
         }
         return true;
      }
   }

   return GTextViewer::onKeyDown(key);
}

bool LCmdConsoleMonitor::onInitMenu ()
{
   bool anySelected = isAnySelectedText();
   bool moreThanOneLine = (getLinesCount() > 1);

   setCommandEnableState("cmdCopy", anySelected);
   setCommandEnableState("cmdSearch", moreThanOneLine);
   setCommandEnableState("cmdSearchNext", moreThanOneLine);
   setCommandEnableState("cmdGotoLineNum", moreThanOneLine);
   setCommandEnableState("cmdSelectAll", moreThanOneLine);

   return true;
}

int LCmdConsoleMonitor::getUserSelectedLineNumber ()
{
   int max = getLinesCount();
   int curLine = getCurrentPosY();
   LCmdDlgGotoLine dlg(1, max, curLine + 1, 0);
   return dlg.execute(this);
}

void LCmdConsoleMonitor::appendTextFromGuiThread ( const GString& format, 
                                                   const GVArgs& args )
{
   GString txt(format, args);
   
   if (txt.length() == 1) // Possibly a "control character".
   {
      switch (txt[0])
      {
         case 0x1A: // Ctrl+Z (End-of-file).
            txt = "^Z\r\n";
            break;
         // TODO: Add support for more control characters here? (e.g. ^C.)
         default:
            break;
      }
   }

   // Make sure the text viewer (console monitor) has at least one line.
   if (getLinesCount() <= 0)
      appendLine(GString::EOL);

   // Assume lineBuff always contains a copy of the last line in the console 
   // monitor. Thus, we can simply update our buffer and just replace the 
   // last line of the console monitor with the current content of the 
   // buffer whenever we have received any more characters from the pipe.
   char prevChr = '\0';
   int lineIndex = getLinesCount() - 1;
   for (int i=0, num=txt.length(); i<num; i++)
   {
      char chr = txt[i];
      if (chr == '\r')
      {
         lineBuff += GString::EOL;
         updateLine(lineIndex, lineBuff);
         lineBuffPos = 0;
      }
      else
      if (chr == '\n')
      {
         if (prevChr != '\r')
         {
            lineBuff += GString::EOL;
            updateLine(lineIndex, lineBuff);
         }
         lineIndex++;
         lineBuffPos = 0;
         lineBuff.clear();
         // Add a new empty line.
         appendLine(GString::EOL); 
      }
      else
      {
         if (lineBuffPos < lineBuff.length())
            lineBuff.setCharAt(lineBuffPos, chr);
         else
            lineBuff += chr;
         lineBuffPos++;
      }
      prevChr = chr;
   }

   // Update the bottom console monitor line with the trailing characters 
   // after the last CR/LF, if any.
   updateLine(lineIndex, lineBuff);

   // Update statusbar cells that shows console line count and active line.
   lcmd->mainWin.updateStatusbarConsoleCurrentLine();
   lcmd->mainWin.updateStatusbarConsoleLineCount();
}

void LCmdConsoleMonitor::appendTextFromSecondaryThread ( GThread& thread,
                                                         const GString& format,
                                                         const GVArgs& args )
{
   GString txt(format, args);
   GUserMessage um("AppendText", txt);
   thread.sendGuiUserMessage(*this, um);
}
