/* --------------------------------------------------------------------------
 *
 * Copyright (C) 2007 Leif Erik Larsen, Kjerringvik, Norway.
 *
 * This file is part of the Open Source Edition of Larsen Commander, as
 * available from http://home.online.no/~leifel/lcmd/.  This code is free 
 * software; you can redistribute it and/or modify it under the terms of 
 * the GNU General Public License version 3 only, as published by the 
 * Free Software Foundation.  
 *
 * This code is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 * version 3 at http://www.gnu.org/licenses/gpl-3.0.txt for more details 
 * (a copy is included in the LICENSE file that accompanied this code).
 *
 * ------------------------------------------------------------------------ */

#include "lcmd/LCmdDlgCalcDirSize.h"
#include "lcmd/LCmdFileItem.h"
#include "lcmd/LCmdFilePanel.h"
#include "lcmd/LCmdFilePanelInfoBar.h"

#include "glib/GProgram.h"
#include "glib/io/GFileNotFoundException.h"
#include "glib/gui/GDialogPanel.h"
#include "glib/gui/GPushButton.h"
#include "glib/vfs/GFile.h"
#include "glib/exceptions/GThreadStartException.h"
#include "glib/util/GStringUtil.h"

LCmdDlgCalcDirSize::Result::Result ()
                           :countFiles(0),
                            countDirs(0),
                            totalSize(0),
                            totalSizeAllocated(0),
                            deepestSubDirLevel(0)
{
}

LCmdDlgCalcDirSize::Result::~Result ()
{
}

LCmdDlgCalcDirSize::LCmdDlgCalcDirSize ( LCmdFilePanel* fpanel, 
                                         GVfs& vfs,
                                         const GString& startDir,
                                         RootEntry& rootEntry,
                                         bool needFileCountOnly )
                   :GWorkerThread("DlgCalcDirSize", 200),
                    fpanel(fpanel),
                    vfs(vfs),
                    curDir(startDir),
                    statusOK(true),
                    finished(false),
                    waitWhenFinished(false),
                    rootEntry(rootEntry),
                    subDirLevel(0),
                    needFileCountOnly(needFileCountOnly)
{
}

LCmdDlgCalcDirSize::~LCmdDlgCalcDirSize ()
{
}

bool LCmdDlgCalcDirSize::calcSubDir ()
{
   subDirLevel += 1;
   if (subDirLevel > res.deepestSubDirLevel)
      res.deepestSubDirLevel = subDirLevel;

   // Use a temporary directory name buffer so that the progress bar thread
   // will not show the "\\*.*" at end of directory when showing content
   // of 'curDir' during progress.
   GString pattern(curDir.length() + 4);
   pattern += curDir;
   vfs.slash(pattern);
   pattern += "*";

   // ---
   GVfs::List list;
   vfs.fillList(list, pattern, true, true, true, true);
   for (int i=0, num=list.size();
        i<num && statusOK;
        i++)
   {
      const GVfs::List::Item& item = list[i]; // Make this a fast one.
      res.totalSize += item.getSizeIdeal();
      res.totalSizeAllocated += item.getSizeAllocated();
      if (item.isDirectory())
      {
         res.countDirs++;
         vfs.slash(curDir);
         curDir += item.getFName();
         statusOK = calcSubDir();
      }
      else
      {
         res.countFiles++;
      }
   }

   // Remove the trailing filename
   int index = curDir.length();
   for (; index > 0 && !vfs.isSlash(curDir[index]); index--);
   if (vfs.isSlash(curDir[index]))
      curDir.cutTailFrom(index);

   subDirLevel -= 1;

   return statusOK;
}

void LCmdDlgCalcDirSize::calcItem ( LCmdFileItem& item, int idxFilePanelItem )
{
   if (item.isUpDir())
      return;
   GString fileName = item.getFileName();
   if (item.isDirectory())
   {
      longlong prevTotalSize = res.totalSize;
      longlong prevTotalSizeAllocated = res.totalSizeAllocated;

      curDir = vfs.getCurrentDirectory(true);
      curDir += fileName;
      if (calcSubDir())
      {
         // Update the item in the file panel.
         int itemIndex = fpanel != null ? fpanel->findItem(fileName) : -1;
         if (itemIndex >= 0)
         {
            LCmdFileItem& itm = fpanel->items[itemIndex];
            itm.fileSize = res.totalSize - prevTotalSize;
            itm.fileSizeAlloc = res.totalSizeAllocated - prevTotalSizeAllocated;
            int flags = itm.getInternalFlags();
            itm.setInternalFlags(flags | LCmdFileItem::FINFO_SHOWSIZE);
            if (fpanel->markedFilesCount > 0)
               fpanel->sizeOfMarkedFiles += itm.fileSize;
         }

         res.countDirs++; // Count this directory it self.
      }
   }
   else
   {
      res.countFiles++; // Count this file it self.
      res.totalSize += item.fileSize;
      res.totalSizeAllocated += item.fileSizeAlloc;
   }

   if (idxFilePanelItem >= 0 && fpanel != null)
   {
      GInteger userData(idxFilePanelItem);
      sendUserMessageToMonitor("UpdtItemSize", &userData);
   }
}

void LCmdDlgCalcDirSize::calcSelectedItems ()
{
   if (fpanel == null)
      return;
   int curSel = fpanel->getCurrentSelectedIndex();
   if (fpanel->markedFilesCount > 0)
   {
      int count = fpanel->items.getCount();
      for (int i=0; i<count; i++)
      {
         LCmdFileItem& item = fpanel->items.get(i); // Make this a fast one
         if (item.isMarked())
            calcItem(item, i);
      }
   }
   else
   if (curSel >= 0)
   {
      LCmdFileItem& item = fpanel->items.get(curSel);
      calcItem(item, curSel);
   }
}

void LCmdDlgCalcDirSize::calcItems ( GArray<LCmdCopyFileItem>& items )
{
   const int count = items.getCount();
   for (int i=0; i<count; i++)
   {
      try {
         LCmdCopyFileItem itm = items[i];
         GString srcPath = itm.getSourcePath();
         LCmdFileItem item(vfs, srcPath);
         calcItem(item);
      } catch (GFileNotFoundException& /*e*/) {
         // Can't find the <i>LCmdFileItem</i>.
         // Not critical by any means, so just ignore it.
      }
   }

   // Request the infobar and the INFO Panel to be updated to show
   // the current number and total size of all (marked) items.
   if (fpanel != null)
   {
      fpanel->infoBar.updateAllFileItemInfoCells();
      fpanel->invalidateAll(true);
      LCmdFilePanel& oppositePanel = fpanel->getOppositePanel();
      oppositePanel.info.updateUponCalcSize();
   }
}

void LCmdDlgCalcDirSize::runTheWorkerThread ( GWorkerThread& worker )
{
   finished = false;
   rootEntry.doIt(*this);
   finished = true;
   if (waitWhenFinished && !isStopRequested())
   {
      sendUserMessageToMonitor("FINISHED");
      waitForOK.reset();
      waitForOK.wait();
   }
}

void LCmdDlgCalcDirSize::onWorkerThreadUpdateMonitor ( GWorkerThread& worker, 
                                                       GDialogPanel& monitor )
{
   if (!finished)
      updateMonitor(monitor);
}

void LCmdDlgCalcDirSize::onWorkerThreadInitDialog ( GWorkerThread& worker, 
                                                    GDialogPanel& monitor )
{
   statusOK = true;
   GString fsysName = vfs.getFileSystemName();
   GStringl str1("%Txt_DlgCalcDirSize_Alloc", GVArgs(fsysName));
   monitor.setComponentValue("121", str1); // "Allocated space (FS is %s)"
}

void LCmdDlgCalcDirSize::updateMonitor ( GDialogPanel& monitor )
{
   monitor.setComponentValue("102", curDir); // Current directory
   monitor.setComponentValue("114", GStringUtil::ToByteCountString(res.countDirs, true)); // # Directories
   monitor.setComponentValue("115", GStringUtil::ToByteCountString(res.countFiles, true)); // # Files
   monitor.setComponentValue("123", GStringUtil::ToByteCountString(res.countDirs + res.countFiles, true)); // SUM
   monitor.setComponentValue("106", GStringUtil::ToByteCountString(res.totalSizeAllocated, false, false, false)); // Total size allocated
   monitor.setComponentValue("107", GStringUtil::ToByteCountString(res.totalSize, false, false, false)); // Total ideal size

   longlong wastage = res.totalSizeAllocated - res.totalSize;
   GString wastageStr = GStringUtil::ToByteCountString(wastage, false, false, false);
   monitor.setComponentValue("109", wastageStr); // Wastage
}

void LCmdDlgCalcDirSize::onWorkerThreadCommand ( GWorkerThread& worker, 
                                                 GDialogPanel& /*monitor*/, 
                                                 const GString& cmd )
{
   if (cmd == "DLG_CANCEL")
   {
      if (finished)
      {
         // The worker thread is waiting for the user to click the OK-
         // button, which the user has done now. So let the worker
         // thread know it.
         waitForOK.notifyAll();
      }
      else
      {
         // The user want to cancel the progress.
         requestStop();
         statusOK = false;
      }
   }
}

void LCmdDlgCalcDirSize::onWorkerThreadUserEvent ( GWorkerThread& worker, 
                                                   GDialogPanel& monitor, 
                                                   const GString& msgID, 
                                                   GObject* userParam )
{
   if (msgID == "UpdtItemSize")
   {
      // This message is sent from {@link #calcItem()}
      // if parameter "idxFilePanelItem" is >= 0.
      // We should invalidate all parts of the file panel that contains
      // size-related information about the indexed filename item.
      GInteger* idxObj = dynamic_cast<GInteger*>(userParam);
      if (idxObj != null && fpanel != null)
      {
         int idx = idxObj->intValue();
         GRectangle itemRect;
         fpanel->calcItemRect(idx, itemRect);
         LCmdFilePanelModeAbstract& view = fpanel->getCurrentView();
         view.invalidateRect(itemRect);
         fpanel->infoBar.updateAllFileItemInfoCells();
      }
   }
   else
   if (msgID == "FINISHED")
   {
      // The worker thread has finished.
      updateMonitor(monitor); // Force update with the final count values.
      if (waitWhenFinished)
      {
         monitor.setComponentVisible("200", false);
         monitor.setComponentText("102", "%Txt_DlgCalcDirSize_Finished");
         GPushButton& butt = dynamic_cast<GPushButton&>(monitor.getComponentByID("DLG_CANCEL"));
         butt.setText("%DLG_OK");
         butt.setIconID("ICON_OK");
      }
   }
}

bool LCmdDlgCalcDirSize::CalcSizeOfAllItems ( LCmdFilePanel* fpanel, 
                                              GVfs& itemsVfs,
                                              GArray<LCmdCopyFileItem>& items, 
                                              longlong* pSize, 
                                              int* pCount )
{
   /**
    * Calculate all the specified LCmdCopyFileItem's.
    *
    * @author  Leif Erik Larsen
    * @since   2004.12.23
    */
   class MyRootEntry : public RootEntry 
   {
      GArray<LCmdCopyFileItem>& items;
      public:
      MyRootEntry ( GArray<LCmdCopyFileItem>& items ) 
         : items(items) 
      {
      }
      virtual void doIt ( LCmdDlgCalcDirSize& obj ) 
      {
         obj.calcItems(items);
      }
   };

   GProgram& prg = GProgram::GetProgram();
   GWindow& mwin = prg.getMainWindow();

   try {
      if (fpanel != null && fpanel->markedFilesCount > 0)
         fpanel->sizeOfMarkedFiles = 0;
      GVfs& vfs = (fpanel != null ? fpanel->vfs.peek() : itemsVfs);
      MyRootEntry rootEntry(items);
      bool needCountOnly = (pSize == null);
      LCmdDlgCalcDirSize calc(fpanel, vfs, GString::Empty, rootEntry, needCountOnly);
      calc.waitWhenFinished = false;
      calc.workModal(mwin);
      if (fpanel != null && fpanel->markedFilesCount > 0)
      {
         fpanel->sizeOfMarkedFiles = calc.res.totalSize;
         fpanel->infoBar.invalidateAll(true);
      }
      if (pSize != null)
         *pSize = calc.res.totalSize;
      if (pCount != null)
         *pCount = calc.res.countFiles + calc.res.countDirs;
      return calc.statusOK;
   } catch (GThreadStartException& e) {
      GString estr = e.toString();
      mwin.showMessageBox(estr, GMessageBox::TYPE_ERROR);
      return false;
   }
}

bool LCmdDlgCalcDirSize::CmdCalcSubDirSize ( LCmdFilePanel& fpanel )
{
   /**
    * Calculate every selected/tagged item in the specified file panel.
    *
    * @author  Leif Erik Larsen
    * @since   2004.12.23
    */
   class MyRootEntry : public RootEntry 
   {
      virtual void doIt ( LCmdDlgCalcDirSize& obj ) 
      {
         obj.calcSelectedItems();
      }
   };

   try {
      if (fpanel.markedFilesCount > 0)
         fpanel.sizeOfMarkedFiles = 0;
      MyRootEntry rootEntry;
      GProgram& prg = GProgram::GetProgram();
      GWindow& mwin = prg.getMainWindow();
      GVfs& vfs = fpanel.vfs.peek();
      LCmdDlgCalcDirSize calc(&fpanel, vfs, GString::Empty, rootEntry, false);
      calc.waitWhenFinished = true;
      calc.workModal(mwin, 0);
      return calc.statusOK;
   } catch (GThreadStartException& e) {
      fpanel.showMessageBox(e.toString(), GMessageBox::TYPE_ERROR);
      return false;
   }
}

bool LCmdDlgCalcDirSize::CmdCalcRootDirSize ( LCmdFilePanel& fpanel )
{
   /**
    * Calculate all parts of the current drive of the specified file panel.
    *
    * @author  Leif Erik Larsen
    * @since   2004.12.23
    */
   class MyRootEntry : public RootEntry 
   {
      virtual void doIt ( LCmdDlgCalcDirSize& obj ) 
      {
         obj.calcSubDir();
      }
   };

   try {
      MyRootEntry rootEntry;
      GProgram& prg = GProgram::GetProgram();
      GWindow& mwin = prg.getMainWindow();
      GVfs& vfs = fpanel.vfs.peek();
      GString startDir = vfs.getRootDir();
      LCmdDlgCalcDirSize calc(&fpanel, vfs, startDir, rootEntry, false);
      calc.waitWhenFinished = true;
      calc.workModal(mwin, 0);
      return calc.statusOK;
   } catch (GThreadStartException& e) {
      fpanel.showMessageBox(e.toString(), GMessageBox::TYPE_ERROR);
      return false;
   }
}
