/* --------------------------------------------------------------------------
 *
 * Copyright (C) 2007 Leif Erik Larsen, Kjerringvik, Norway.
 *
 * This file is part of the Open Source Edition of Larsen Commander, as
 * available from http://home.online.no/~leifel/lcmd/.  This code is free 
 * software; you can redistribute it and/or modify it under the terms of 
 * the GNU General Public License version 3 only, as published by the 
 * Free Software Foundation.  
 *
 * This code is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 * version 3 at http://www.gnu.org/licenses/gpl-3.0.txt for more details 
 * (a copy is included in the LICENSE file that accompanied this code).
 *
 * ------------------------------------------------------------------------ */

#include "lcmd/LCmdDlgQuickDirList.h"
#include "lcmd/LCmdFilePanel.h"

#include "glib/GProgram.h"
#include "glib/gui/GDialogPanel.h"
#include "glib/gui/GPushButton.h"
#include "glib/gui/event/GDialogMessage.h"
#include "glib/gui/GListBox.h"
#include "glib/gui/GUserStringDlg.h"

LCmdDlgQuickDirList::LCmdDlgQuickDirList ()
                    :storedDirs(10)
{
   // Make sure that the array of stored directories has room for
   // 10 items from the start so that we can always use the set() method
   // instead of add() to update the content of this array.
   for (int i=0; i<10; i++)
      storedDirs.add(new GString(""));
}

LCmdDlgQuickDirList::~LCmdDlgQuickDirList () 
{
}

int LCmdDlgQuickDirList::execute ( GWindow& parentWin )
{
   GProgram& prg = GProgram::GetProgram();
   GString res = prg.executeDialog(&parentWin, "DlgQuickDirList", this);
   try {
      int idx = GInteger::ParseInt(res);
      if (idx < 0 || idx > 9)
         return -1; // Cancel was selected, so return -1.
      else
         return idx;
   } catch (GNumberFormatException& /*e*/) {
      // Should never happen, but in case.
      return -1;
   }
}

int LCmdDlgQuickDirList::add ( LCmdFilePanel& panel, int index )
{
   if (index < 0 || index > 9)
   {
      for (index=0; index<10; index++)
         if (storedDirs[index] == "")
            break;

      if (index > 9)
         return -1; // No more free directory items
   }

   GString dir = panel.getCurrentSysDirectory(false);
   storedDirs.set(index, new GString(dir));
   return index;
}

bool LCmdDlgQuickDirList::retrieve ( LCmdFilePanel& panel, int index )
{
   if (index < 0 || index > 9)
      return false;
   else
   if (storedDirs[index] != "")
      return panel.walkDir(storedDirs[index]);
   else
      return false;
}

bool LCmdDlgQuickDirList::pickFromDialog ( LCmdFilePanel& panel )
{
   GProgram& prg = GProgram::GetProgram();
   GWindow& mwin = prg.getMainWindow();
   int idx = execute(mwin);
   if (idx >= 0)
      return retrieve(panel, idx);
   else
      return false;
}

void LCmdDlgQuickDirList::loadFromProfile ( GSectionBag& ini, const GString& sectName )
{
   for (int i=0; i<10; i++)
   {
      GString itemName("%02d", GVArgs(i+1));
      GString dir = ini.getString(sectName, itemName, "");
      storedDirs.set(i, new GString(dir));
   }
}

void LCmdDlgQuickDirList::writeToProfile ( GSectionBag& ini, const GString& sectName )
{
   for (int i=0; i<10; i++)
   {
      GString itemName("%02d", GVArgs(i+1));
      ini.putString(sectName, itemName, storedDirs[i], true);
   }
}

void LCmdDlgQuickDirList::onListBoxSelectionChanged ( class GAbstractListBox& abstractListb ) 
{
}

void LCmdDlgQuickDirList::onListBoxSelectionDblClick ( class GAbstractListBox& abstractListb ) 
{
}

void LCmdDlgQuickDirList::onListBoxSelectionChangedByCharKey ( GAbstractListBox& abstractListb )
{
   GListBox& listb = dynamic_cast<GListBox&>(abstractListb);
   GDialogPanel& dlg = *listb.getOwnerDialogPanel();
   const GString& listbID = listb.getName();
   dlg.sendDialogMessage(GM_LISTBOX_DBLCLICK, listbID);
}

bool LCmdDlgQuickDirList::handleDialogMessage ( GDialogMessage& msg )
{
   GDialogPanel& dlg = msg.getDialog();
   switch (msg.getID())
   {
      case GM_INITDIALOG: 
      {
         GListBox& list = dynamic_cast<GListBox&>(dlg.getComponentByID("101"));
         list.addListBoxSelectionListener(this);

         // Fill the directories into the list and try to automatically
         // select the item which is the current directory.
         int index = 0;
         LCmdFilePanel& cp = LCmdFilePanel::GetCurrentPanel();
         GString curDir = cp.getCurrentSysDirectory(false);
         for (int i=0; i<10; i++)
         {
            const GString& dir = storedDirs[i];
            GString dirText("%d:  %s", GVArgs(i).add(dir));
            list.addItem(dirText, "IDP_DIRDIR");
            if (dir.equalsIgnoreCase(curDir))
               index = i;
         }

         list.setSelectedIndex(index);
         return true; 
      }

      case GM_CTRLCHANGED:
      {
         GString ctrlID = msg.getParam1String();
         if (ctrlID == "101") // User has selected an item
         {
            GListBox& list = dynamic_cast<GListBox&>(dlg.getComponentByID("101"));
            int index = list.getSelectedIndex();
            int num = list.getItemCount();
            dlg.setComponentEnabled("DLG_UP", num > 0 && index > 0);
            dlg.setComponentEnabled("DLG_DOWN", num > 0 && index >= 0 && index < num-1);
            bool enableOK = false;
            if (index >= 0)
            {
               GString selectedTxt = list.getItemText(index).substring(4);
               if (selectedTxt != "")
                  enableOK = true;
            }
            dlg.setComponentEnabled("DLG_OK", enableOK);
            if (enableOK)
               dlg.setDefaultPushButton(&dynamic_cast<GPushButton&>(dlg.getComponentByID("DLG_OK")));
            else
               dlg.setDefaultPushButton(&dynamic_cast<GPushButton&>(dlg.getComponentByID("DLG_EDIT")));
         }
         return true; 
      }

      case GM_LISTBOX_DBLCLICK: 
      {
         GString ctrlID = msg.getParam1String();
         if (ctrlID == "101") // User has double clicked on an item.
         {
            GListBox& list = dynamic_cast<GListBox&>(dlg.getComponentByID("101"));
            int index = list.getSelectedIndex();
            if (index >= 0)
            {
               GString itemText = list.getItemText(index);
               GString txt = itemText.substring(4);
               if (txt == "")
                  dlg.sendDialogMessage(GM_COMMAND, "DLG_EDIT");
               else
                  dlg.sendDialogMessage(GM_COMMAND, "DLG_OK");
            }
         }
         return true; 
      }

      case GM_COMMAND: 
      {
         GListBox& list = dynamic_cast<GListBox&>(dlg.getComponentByID("101"));
         GString cmdID = msg.getParam1String();
         if (cmdID == "DLG_EDIT")
         {
            int index = list.getSelectedIndex();
            if (index >= 0)
            {
               GString txt = list.getItemText(index);
               GString def = txt.substring(4);
               GString titleStr = "%Txt_DlgQuickDirList_EditDlgTitle";
               GString leaderStr = "%Txt_DlgQuickDirList_EditDlgLeader";
               GString dir = GUserStringDlg::Get(&dlg, titleStr, leaderStr, def, GString::Empty, GFile::MAXPATH);
               if (dir != "")
               {
                  GString newText("%d:  %s", GVArgs(index).add(dir));
                  list.setItemText(index, newText);
                  list.sendCtrlChanged();
               }
            }
            dlg.setComponentFocus("101");
         }
         else
         if (cmdID == "DLG_CLEAR") // Clear current item only.
         {
            int index = list.getSelectedIndex();
            if (index >= 0 && index <= 9)
            {
               GString newStr("%d:  ", GVArgs(index));
               list.setItemText(index, newStr);
               list.sendCtrlChanged();
            }
            dlg.setComponentFocus("101");
         }
         else
         if (cmdID == "DLG_UP")
         {
            int index = list.getSelectedIndex();
            if (index > 0)
            {
               GString txt = list.getItemText(index);
               txt.setCharAt(0, char('0' + index - 1));
               list.removeItem(index);
               list.insertItem(index-1, txt, "IDP_DIRDIR");
               GString txt2 = list.getItemText(index);
               txt2.setCharAt(0, char(txt2[0] + 1));
               list.setItemText(index, txt2);
               list.setSelectedIndex(index-1);
            }
            dlg.setComponentFocus("101");
         }
         else
         if (cmdID == "DLG_DOWN")
         {
            int numItems = list.getItemCount();
            int index = list.getSelectedIndex();
            if (index >= 0 && index < numItems - 1)
            {
               GString txt = list.getItemText(index);
               txt.setCharAt(0, char('0' + index + 1));
               list.removeItem(index);
               list.insertItem(index+1, txt, "IDP_DIRDIR");
               GString txt2 = list.getItemText(index);
               txt2.setCharAt(0, char(txt2[0] - 1));
               list.setItemText(index, txt2);
               list.setSelectedIndex(index+1);
            }
            dlg.setComponentFocus("101");
         }
         else
         if (cmdID == "DLG_CLOSE") // Close
         {
            for (int i=0; i<10; i++)
            {
               GString txt = list.getItemText(i).substring(4);
               storedDirs.set(i, new GString(txt));
            }
            dlg.dismiss(GInteger::ToString(-1));
         }
         else
         if (cmdID == "DLG_OK")
         {
            for (int i=0; i<10; i++)
            {
               GString txt = list.getItemText(i).substring(4);
               storedDirs.set(i, new GString(txt));
            }
            int curIndex = list.getSelectedIndex();
            dlg.dismiss(GInteger::ToString(curIndex));
         }
         else
         if (cmdID == "DLG_CANCEL") // Dismiss without updating the history list.
         {
            dlg.dismiss(GInteger::ToString(-1));
         }
         return true; 
      }

      default:
         return false;
   }
}

