/* --------------------------------------------------------------------------
 *
 * Copyright (C) 2007 Leif Erik Larsen, Kjerringvik, Norway.
 *
 * This file is part of the Open Source Edition of Larsen Commander, as
 * available from http://home.online.no/~leifel/lcmd/.  This code is free 
 * software; you can redistribute it and/or modify it under the terms of 
 * the GNU General Public License version 3 only, as published by the 
 * Free Software Foundation.  
 *
 * This code is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 * version 3 at http://www.gnu.org/licenses/gpl-3.0.txt for more details 
 * (a copy is included in the LICENSE file that accompanied this code).
 *
 * ------------------------------------------------------------------------ */

#include "lcmd/LCmdExitWorker.h"
#include "lcmd/LCmdCmdLineEntry.h"
#include "lcmd/LCmdOptions.h"
#include "lcmd/LCmdProcess.h"

#include "glib/gui/GDialogPanel.h"
#include "glib/gui/GProgressBar.h"

LCmdExitWorker::LCmdExitWorker ()
               :GWorkerThread("DlgExitWorker", // dlgResourceID
                              0, // monitorUpdtMillis
                              null, // adapter
                              false, // autoDeleteAdapter
                              "LCmdChildProcessStopperThread"), // threadName
                entry(LCmdCmdLineEntry::GetCmdLineEntry()),
                processCountStart(0),
                detach(false),
                ignore(false)
{
}

LCmdExitWorker::~LCmdExitWorker ()
{
}

void LCmdExitWorker::onWorkerThreadInitDialog ( GWorkerThread& worker, 
                                                GDialogPanel& monitor )
{
   monitor.setComponentEnabled("DLG_CANCEL", true);
   monitor.setComponentText("NowStopping", GString::Empty);
   GProgressBar& pbar = dynamic_cast<GProgressBar&>(monitor.getComponentByID("Progress"));
   countStopped = 0;
   pbar.setMinValue(countStopped);
   pbar.setCurrentValue(countStopped);
   processCountStart = entry.getRunningChildrenCount();
   pbar.setMaxValue(processCountStart);
}

void LCmdExitWorker::onWorkerThreadCommand ( GWorkerThread& worker,
                                             GDialogPanel& monitor, 
                                             const GString& cmdID )
{
   if (cmdID == "DLG_CANCEL")
   {
      monitor.setComponentEnabled(cmdID, false);
      worker.requestStop(false);
   }
}

void LCmdExitWorker::onWorkerThreadUpdateMonitor ( GWorkerThread& worker,
                                                   GDialogPanel& monitor )
{
   GProgressBar& pbar = dynamic_cast<GProgressBar&>(monitor.getComponentByID("Progress"));
   pbar.setCurrentValue(countStopped);
   monitor.setComponentText("NowStopping", nowStopping);
}

void LCmdExitWorker::runTheWorkerThread ( GWorkerThread& worker )
{
   for (int i=processCountStart-1; i>=0 && !worker.isStopRequested(); i--)
   {
      LCmdProcessLauncher& child = entry.runningChildren[i];
      nowStopping = child.getProgramName();
      worker.updateMonitor();

      if (detach || ignore)
      {
         child.detachChildProg();
      }
      else
      {
         bool ok = child.breakChildProg(GSystem::BT_Exit, true);
         if (!ok && !worker.isStopRequested())
            child.breakChildProg(GSystem::BT_Kill, true);
      }

      if (worker.isStopRequested())
         break;

      // Wait until the child process has been deregistered from LCmd.
      if (!ignore)
      {
         while (!worker.isStopRequested() && countStopped < processCountStart)
         {
            if (child.isRunning())
            {
               GThread::Sleep(250);
               continue;
            }
            countStopped++;
            worker.updateMonitor();
         }
      }
   }

   if (!worker.isStopRequested())
      worker.updateMonitor();

   // Wait some millis, to give the user a chance to see that all 
   // child processes has been detached or stopped.
   if (!ignore && processCountStart > 0)
      GThread::Sleep(500);
}

void LCmdExitWorker::performExit ()
{
   detach = false;
   ignore = false;

   GProgram& prg = GProgram::GetProgram();
   GFrameWindow& mwin = dynamic_cast<GFrameWindow&>(prg.getMainWindow());
   LCmdOptions& opt = LCmdOptions::GetOptions();

   // Set the "Exit..." text in the statusbar.
   mwin.pushStatusbarText("%StrStatusbar_Status_Exit");

   if (entry.getRunningChildrenCount() > 0)
   {
      GStringl msg("%Txt_Warning_ExitWhenRunningChildren");
      GMessageBox::Answer answ;
      answ = mwin.showMessageBox(msg, GMessageBox::TYPE_QUESTION, 
                                 "(U1)(u2)(u3)c",  GString::Empty,  false, 
                                 "%Txt_Warning_ExitWhenRunningChildren_Detach",
                                 "%Txt_Warning_ExitWhenRunningChildren_Kill",
                                 "%Txt_Warning_ExitWhenRunningChildren_Ignore");
      if (answ == GMessageBox::IDCANCEL)
      {
         mwin.popStatusbarText();
         return;
      }

      if (answ == GMessageBox::IDUSER1) // "Detach" button.
         detach = true;
      else
      if (answ == GMessageBox::IDUSER2) // "Kill" button.
         detach = false;
      else
      if (answ == GMessageBox::IDUSER3) // "Ignore" button.
         ignore = true;
   }

   if (!ignore && opt.confirm.prgExit)
   {
      GStringl msg("%Txt_Warning_ConfirmExit");
      GMessageBox::Answer answ;
      answ = mwin.showMessageBox(msg, GMessageBox::TYPE_QUESTION, "Ync");
      if (answ != GMessageBox::IDYES)
      {
         mwin.popStatusbarText();
         return;
      }
   }

   if (workModal(mwin))
      mwin.postMessage(WM_QUIT);
   else
      mwin.popStatusbarText();
}
