/* --------------------------------------------------------------------------
 *
 * Copyright (C) 2007 Leif Erik Larsen, Kjerringvik, Norway.
 *
 * This file is part of the Open Source Edition of Larsen Commander, as
 * available from http://home.online.no/~leifel/lcmd/.  This code is free 
 * software; you can redistribute it and/or modify it under the terms of 
 * the GNU General Public License version 3 only, as published by the 
 * Free Software Foundation.  
 *
 * This code is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 * version 3 at http://www.gnu.org/licenses/gpl-3.0.txt for more details 
 * (a copy is included in the LICENSE file that accompanied this code).
 *
 * ------------------------------------------------------------------------ */

#ifndef __LCMD_PANELFILESCOLUMNSBAR
#define __LCMD_PANELFILESCOLUMNSBAR

#include "glib/gui/GToolbar.h"
#include "glib/gui/GToolbarButton.h"
#include "glib/gui/event/GCommandMap.h"
#include "lcmd/LCmdFilePanelSortOptions.h"

/**
 * This is the Columns Bar of a file panel.
 *
 * The Column Bar window will be visible in the FULL mode only.
 *
 * @author  Leif Erik Larsen
 * @since   1998.04.08
 */
class LCmdFilePanelColumnsBar : public GToolbar
{
   public:

      /** This is the base class of all columnbar buttons. */
      class Button : public GToolbarButton
      {
         protected:

            class LCmdFilePanelColumnsBar& clmnBar;

         private:

            int customizedWidth;
            GString shortcutKeyCmdID;

         private:

            LCmdFilePanelSortOptions::SORT_WHAT sortID;

         protected:

            Button ( const GString& id, 
                     const GString& shortcutKeyCmdID, 
                     class LCmdFilePanelColumnsBar& clmnBar, 
                     LCmdFilePanelSortOptions::SORT_WHAT sortID );

         private:

            bool isAtLeft ( int xpos, int ypos ) const;
            bool isAtRight ( int xpos, int ypos ) const;

         protected:

            int getIconWidth () const;

            /**
             * @see GToolbarButton#isMousePosOverridden
             */
            bool isMousePosOverridden ( int xpos, int ypos ) const;

            bool onButton2Click ( int xpos, int ypos, const GWindowMessage::InputFlags& flags );

            bool onButton1Down ( int xpos, int ypos, const GWindowMessage::InputFlags& flags );

            /**
             * @return True if and only if the mouse cursor was actually set.
             */
            bool onMouseMove ( int xpos, int ypos, const GWindowMessage::InputFlags& flags );

         public:

            int getMinimumWidth () const;
            void setCustomizedWidth ( int width, bool updateClmnsbar = true );
            virtual void queryProfile ( const GString& sectName );
            virtual void writeProfile ( const GString& sectName, bool force = false );
            virtual int getPreferredWidth () const;
            virtual const class GIcon* getIcon () const;
            virtual GString getTooltipText ();
            virtual GString getKeyboardShortcutKeyText () const;
            virtual int calcDefaultWidth () const = 0;
            virtual void drawCell ( class GGraphics& g, class LCmdFilePanel& fp, class LCmdFileItem& fitem, const class GRectangle& r, const class GColor& bc, const class GColor& fc, bool isCurSelected ) = 0;
            virtual void setVisible ( bool flag = true );

         public:

            void cmdColumnsbarButtonHide ( class GAbstractCommand* cmd = null );
            void cmdColumnsbarButtonDefWidth ( class GAbstractCommand* cmd = null );
            void cmdColumnsbarButtonMinWidth ( class GAbstractCommand* cmd = null );

         DECLARE_COMMAND_TABLE(Button);
      };

   private:

      /** The button for the Icon-column. */
      class ButtonIcon : public Button
      {
         public:

            ButtonIcon ( class LCmdFilePanelColumnsBar& clmnBar );
            int getPreferredWidth () const;
            int calcDefaultWidth () const;
            void drawCell ( class GGraphics& g, class LCmdFilePanel& fp, class LCmdFileItem& fitem, const class GRectangle& r, const class GColor& bc, const class GColor& fc, bool isCurSelected );
            virtual const GIcon* getIcon () const;
      };

      /** The button for the Filename-column. */
      class ButtonName : public Button
      {
         public:

            ButtonName ( class LCmdFilePanelColumnsBar& clmnBar );
            int calcDefaultWidth () const;
            void drawCell ( class GGraphics& g, class LCmdFilePanel& fp, class LCmdFileItem& fitem, const class GRectangle& r, const class GColor& bc, const class GColor& fc, bool isCurSelected );
      };

      /** The button for the Extension-column. */
      class ButtonExt : public Button
      {
         public:

            ButtonExt ( class LCmdFilePanelColumnsBar& clmnBar );
            int calcDefaultWidth () const;
            void drawCell ( class GGraphics& g, class LCmdFilePanel& fp, class LCmdFileItem& fitem, const class GRectangle& r, const class GColor& bc, const class GColor& fc, bool isCurSelected );
      };

      /** The button for the Size-column. */
      class ButtonSize : public Button
      {
         public:

            bool alwaysLeftAlign;

            ButtonSize ( class LCmdFilePanelColumnsBar& clmnBar );
            int calcDefaultWidth () const;
            void drawCell ( class GGraphics& g, class LCmdFilePanel& fp, class LCmdFileItem& fitem, const class GRectangle& r, const class GColor& bc, const class GColor& fc, bool isCurSelected );
      };

      /** The button for the Date-column. */
      class ButtonDate : public Button
      {
         public:

            ButtonDate ( class LCmdFilePanelColumnsBar& clmnBar );
            int calcDefaultWidth () const;
            void drawCell ( class GGraphics& g, class LCmdFilePanel& fp, class LCmdFileItem& fitem, const class GRectangle& r, const class GColor& bc, const class GColor& fc, bool isCurSelected );
      };

      /** The columnsbar button for the Time-column. */
      class ButtonTime : public Button
      {
         public:

            ButtonTime ( class LCmdFilePanelColumnsBar& clmnBar );
            int calcDefaultWidth () const;
            void drawCell ( class GGraphics& g, class LCmdFilePanel& fp, class LCmdFileItem& fitem, const class GRectangle& r, const class GColor& bc, const class GColor& fc, bool isCurSelected );
      };

   friend class Button;

   public:

      /** A reference to the list of which we are assosiated. */
      class LCmdFilePanelModeFull& listWin;

      /** True if we shall not read/write from/to the user profile (ini-file). */
      bool ignoreIniProfile;

      ButtonIcon clmnIcon;
      ButtonName clmnName;
      ButtonExt clmnExt;
      ButtonSize clmnSize;
      ButtonDate clmnDate;
      ButtonTime clmnTime;

      /** 
       * By default, this vector contains { 0, 1, 2, 3, ...n }, where n is 
       * the number of column buttons in the columns bar. This means that 
       * each column appears on screen in the same horizontal order (from 
       * left to right) as of which they was added.
       *
       * However, the columns bar supports changing the column order on
       * screen. This is done with method {@link #setColumnPositionOnScreen}.
       * The column order on screen is maintained by this vector.
       *
       * @author  Leif Erik Larsen
       * @since   2006.01.27
       */
      GVector<int> customColumnSuccession;

   private:

      const class GIcon* iconAscending;
      const class GIcon* iconDescending;
      const class GIcon* iconDummy;

   public:

      LCmdFilePanelColumnsBar ( class LCmdFilePanelModeFull& listWin, 
                                const GString& constraints );

      virtual ~LCmdFilePanelColumnsBar ();

   protected:

      virtual void queryProfile ( const GString& sectName );
      virtual void writeProfile ( const GString& sectName, bool force = false );
      virtual bool onButton2Click ( int xpos, int ypos, const GWindowMessage::InputFlags& flags );

      /**
       * We override this method in order to always keep the 
       * {@link #customColumnSuccession} vector in synch with the 
       * number of columns.
       *
       * @author  Leif Erik Larsen
       * @since   2006.01.27
       */
      virtual class GToolbarElement* addElementImpl ( class GToolbarElement* elm, bool autoDelete = true );

   public:

      virtual int getPreferredHeight () const;

      /**
       * We override this method in order to respect the custom column 
       * succession on screen, as set with call(s) 
       * to {@link #setColumnPositionOnScreen}.
       *
       * @author  Leif Erik Larsen
       * @since   2006.01.27
       */
      virtual void layout ();

      /**
       * Move all the buttons on the columns bar the specified number
       * of horizontal pixels. If the specified pixel count is negative
       * the buttons will be moved to the left, or else they will be moved
       * to the right.
       */
      void moveButtonsHor ( int movex );

      /**
       * The buttons must call this method whenever there has been done
       * some changes that requires all parts of the columnsbar to be
       * updated, including the associated file panel. This is, for instance,
       * when one of the buttons has been resized or its visibility has
       * been toggled.
       */
      void updateTheColumnsbar ();

      void cmdColumnsbarButtonShowAll ( class GAbstractCommand* cmd = null );
      void cmdColumnsbarButtonDefWidthAll ( class GAbstractCommand* cmd = null );
      void cmdColumnsbarButtonMinWidthAll ( class GAbstractCommand* cmd = null );
      int getColumnsCount ();
      Button& getColumnButton ( int index );

      /**
       * Change the horizontal position of the column on screen.
       * The default (if this method is never used) is that the columns 
       * appears on screen in the same order (from left to right) as of 
       * which they was added.
       *
       * @author  Leif Erik Larsen
       * @since   2006.01.27
       * @param   currentIdx Index of which column to move, in screen model.
       * @param   newIdx     New position index of the column, in screen model.
       * @throws  GIllegalArgumentException If the specified index(es) are 
       *                     invalid. E.g. if either one of them is outside 
       *                     valid range.
       */
      void setColumnPositionOnScreen ( int currentIdx, int newIdx );

   DECLARE_COMMAND_TABLE(LCmdFilePanelColumnsBar);
};

#endif // #ifndef __LCMD_PANELFILESCOLUMNSBAR
