/* --------------------------------------------------------------------------
 *
 * Copyright (C) 2007 Leif Erik Larsen, Kjerringvik, Norway.
 *
 * This file is part of the Open Source Edition of Larsen Commander, as
 * available from http://home.online.no/~leifel/lcmd/.  This code is free 
 * software; you can redistribute it and/or modify it under the terms of 
 * the GNU General Public License version 3 only, as published by the 
 * Free Software Foundation.  
 *
 * This code is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 * version 3 at http://www.gnu.org/licenses/gpl-3.0.txt for more details 
 * (a copy is included in the LICENSE file that accompanied this code).
 *
 * ------------------------------------------------------------------------ */

#include "lcmd/LCmdFilePanelModeBrief.h"
#include "lcmd/LCmdFilePanel.h"
#include "lcmd/LCmdFilePanelHeader.h"
#include "lcmd/LCmdFilePanelInfoBar.h"
#include "lcmd/LCmdFilePanelFrame.h"
#include "lcmd/LCmdFileColumn.h"
#include "lcmd/LCmd.h"

#include "glib/GProgram.h"
#include "glib/util/GIsHereTracker.h"
#include "glib/util/GMath.h"

LCmdFilePanelModeBrief::LCmdFilePanelModeBrief ( LCmdFilePanel& fpanel,
                                                 const GString& winName )
                       :GDecoratedWindow(winName,
                                         GString::Empty, 
                                         &fpanel.tabs,
                                         GString::Empty,
                                         WS_VISIBLE,
                                         WS2_OS2Y),
                        LCmdFilePanelModeAbstract(fpanel),
                        fpanel(fpanel),
                        columns(200, -2),
                        curSelectedItem(-1), // Initially no selected item.
                        firstVisibleItem(-1), // Initially no items in list.
                        itemsInListVer(1), // Prevent initial 'divide by zero'.
                        leftMostColumnXPos(0),
                        leftMostColumn(0),
                        itemHeight(0),
                        iconSize(0),
                        hasEverBeenVisible(false)
{
   fpanel.items.addFileItemContainerListener(this);

   setHScrollVisible(true);
   setVScrollVisible(false);

   // Activate the popup menu of which to display when user
   // right-click on a file item or on some other space of this window.
   GProgram& prg = GProgram::GetProgram();
   setPopupMenu("FileItemContextMenu", prg.isUseFancyPopupMenues());

   // Let the file panel object handle the drag-n-drop events.
   setDragDropHandler(&fpanel, false);
}

LCmdFilePanelModeBrief::~LCmdFilePanelModeBrief ()
{
   fpanel.items.removeFileItemContainerListener(this);
}

bool LCmdFilePanelModeBrief::isHorizontallyScrollable () const
{
   return columns.getCount() > 1;
}

void LCmdFilePanelModeBrief::invalidateAll ( bool inclChildren ) const
{
   GDecoratedWindow::invalidateAll(inclChildren);
}

void LCmdFilePanelModeBrief::invalidateRect ( const GRectangle& rect ) const
{
   GDecoratedWindow::invalidateRect(rect);
}

void LCmdFilePanelModeBrief::allItemsHaveBeenRemoved ()
{
   leftMostColumn = 0;
   leftMostColumnXPos = 0;
   curSelectedItem = -1;
   firstVisibleItem = -1;
}

void LCmdFilePanelModeBrief::onViewHasBeenActivated ()
{
   int curSel = curSelectedItem;
   leftMostColumn = 0;
   leftMostColumnXPos = 0;
   layout();
   fpanel.frameWin.layout();
   fpanel.frameWin.headerWin.layout();
   updateHScrollBarRange(); // Will call updateScrollBarPos() as well
   curSelectedItem = -1; // Just to force reselection of current item
   fpanel.selectItem(curSel);
}

int LCmdFilePanelModeBrief::getCurrentSelectedIndex () const
{
   return curSelectedItem;
}

int LCmdFilePanelModeBrief::getFirstVisibleIndex () const
{
   return firstVisibleItem;
}

int LCmdFilePanelModeBrief::calcItemIdxFromPos ( int xpos, int ypos ) const
{
   GRectangle rect(filesRect); // Working copy of current files area.

   if (xpos < 0 ||
       xpos >= rect.x + rect.width - 1 ||
       ypos < 0 ||
       ypos >= rect.y + rect.height - 1)
   {
      // Specified position is outside window, so we must return -1
      return -1;
   }

   int clm;
   int clmXPos = leftMostColumnXPos;
   int clmCount = columns.getCount();
   for (clm = leftMostColumn;
        clm < clmCount;
        clm++)
   {
      clmXPos += columns.get(clm).width;
      if (clmXPos >= xpos)
         break;
   }

   int idx = (rect.y + rect.height - 1 - ypos) / itemHeight;
   if (idx < itemsInListVer)
      idx += clm * itemsInListVer;
   else
      idx = -1;

   if (idx >= fpanel.items.getCount())
      idx = -1;

   return idx >= 0 ? idx : -1;
}

bool LCmdFilePanelModeBrief::calcItemRect ( int itemIndex, GRectangle& rect ) const
{
   rect.clear();

   // Be sure not to use an illegal index (in case there are no items in list).
   if (itemIndex < 0 || itemIndex >= fpanel.items.getCount())
      return false;

   // Get a copy of current panel window size
   GRectangle r = getWindowRect();

   int winHeight = r.height;
   int winWidth = r.width;
   int iColumn = itemIndex / itemsInListVer;
   r.y = winHeight - (((itemIndex % itemsInListVer) + 1) * itemHeight);
   int iXPos = calcColumnXPos(iColumn);
   r.x = iXPos;
   r.height = itemHeight;
   r.width = columns.get(iColumn).width;

   // Don't paint items outside visible area
   if (r.x >= winWidth || r.y <= -itemHeight)
      return false;

   rect = r;
   return true;
}

int LCmdFilePanelModeBrief::calcHScrollPos ()
{
   if (leftMostColumn >= columns.getCount())
      return 0;
   int ret = columns.get(leftMostColumn).xpos - leftMostColumnXPos;
   return ret;
}

void LCmdFilePanelModeBrief::updateHScrollBarPos ()
{
   int pos = calcHScrollPos();
   setHScrollPos(pos);
}

void LCmdFilePanelModeBrief::updateHScrollBarRange ()
{
   int thumbLen = 0;
   int scrollLen = 0;

   if (columns.getCount() > 0)
   {
      thumbLen  = filesRect.width;
      for (int i=0, count=columns.getCount(); i<count; i++)
      {
         int columnWidth = columns.get(i).width;
         scrollLen += columnWidth;
      }
      scrollLen -= thumbLen;
   }

   // Some special logic here in order to prevent recursive calling 
   // of {@link #layout} in case showing a scrollbar cause the same 
   // scrollbar not actually to be needed and vice verca. The point 
   // is that when we show or hide the horizontal scrollbar it may 
   // happen that the re-arrengement of our variable-width columns 
   // ends up with an arrangement that does not match the just-set 
   // scrollbar visibility. In order to prevent this, we use the 
   // Static Is-Here-mechanism. The only negative side-effect of 
   // this is that the user may find that the horizontal scrollbar
   // in some rare cases may be visible even if it doesn't need 
   // to, or invisible when it is actually needed. But this happens 
   // so rarely it is no problem. The problem doing nothing here to 
   // handle this would be much more serious, and could cause the 
   // file items to be painted at corrupt positions, etc. 
   // This was the fact in Larsen Commander versions prior 
   // to version 1.6 beta #5.
   static bool IsHere = false;
   bool dontLayout = IsHere;
   GIsHereTracker isHereTracker(IsHere, true);
   setHScrollRange(scrollLen, thumbLen, dontLayout);
   setVScrollRange(0, 0, dontLayout);

   // Also update the current scrollbar(s) thumb position, so that they also
   // always are placed at the correct position. (E.g. when the file list
   // window has been resized.)
   updateHScrollBarPos();
}

void LCmdFilePanelModeBrief::calcAllColumns ()
{
   leftMostColumn = 0;
   leftMostColumnXPos = 0;

   // Calculate how many columns we need horizontally.
   int numItems = fpanel.items.getCount();
   int numColumns = numItems / itemsInListVer;
   if (numItems % itemsInListVer)
      numColumns += 1;
   if (numColumns <= 0)
      numColumns = 1; // Must always have at least one column

   // Create all the needed Column Objects and calculate their 
   // respective width and x-position.
   columns.removeAll();
   for (int i=0, itemIdx=0; i<numColumns; i++)
   {
      LCmdFileColumn* clmn = new LCmdFileColumn();
      columns.add(clmn);
      int widestColumnItem = 0;
      for (int ii=0; ii<itemsInListVer && itemIdx<numItems; ii++)
      {
         LCmdFileItem& itm = fpanel.items[itemIdx++];
         int width = itm.getBriefWidth();
         if (width > widestColumnItem)
            widestColumnItem = width;
      }

      clmn->width = widestColumnItem + 15;
      if (i == 0)
      {
         clmn->xpos = 0;
      }
      else
      {
         LCmdFileColumn& clmn_left = columns.get(i - 1);
         clmn->xpos = clmn_left.xpos + clmn_left.width;
      }
   }

   // Update the scrollbars of the fileslist of this panel.
   updateHScrollBarRange();
}

void LCmdFilePanelModeBrief::calcItems ()
{
   LCmdOptions& opt = LCmdOptions::GetOptions();
   const int addToItemHeight = opt.various.addToItemHeight;

   // Query pixels height of the font (the dummy text will not be painted)
   itemHeight = 2 + getHeightOfString("X") + addToItemHeight;
   iconSize = fpanel.calcIconSize();

   if (itemHeight < iconSize + 4 + addToItemHeight)
      itemHeight = iconSize + 4 + addToItemHeight;

   if (itemHeight < 6)
      itemHeight = 6;

   // Calculate how many items there are room for vertically
   itemsInListVer = filesRect.height / itemHeight;
   if (itemsInListVer < 1)
      itemsInListVer = 1;  // This item must never be less than one!

   calcAllColumns(); // Will also call {@link #updateHScrollBarRange}.
}

int LCmdFilePanelModeBrief::calcColumnXPos ( int idx ) const
{
   int count = columns.getCount();
   if (idx >= count)
      return 0;

   int x1 = columns.get(idx).xpos;
   int x2 = leftMostColumn >= count ? 0 : columns.get(leftMostColumn).xpos;
   int x3 = leftMostColumnXPos;
   int ret = x1 - x2 + x3;
   return ret;
}

int LCmdFilePanelModeBrief::calcLastVisibleItemNC ()
{
   // Find the rightmost column that is somewhat visible, 
   // and keep its x-position and width.
   GDimension dim = getWindowSize();
   int columnIdx = leftMostColumn;
   int columnXPos = calcColumnXPos(columnIdx);
   int columnWidth = columns.get(columnIdx).width;
   int columnCount = columns.getCount();
   while (columnXPos < dim.width && columnIdx < columnCount - 1)
   {
      columnIdx += 1;
      columnXPos += columnWidth;
      columnWidth = columns.get(columnIdx).width;
      if (columnXPos + columnWidth >= dim.width)
         break;
   }

   // Use the left neightbour column (if any) if the rightmost visible 
   // column is clipped and the left neightbour is not clipped. If both
   // the columns are clipped then use the rightmost visible anyway.
   if (columnIdx - 1 >= 0 && 
         columnXPos + columnWidth > dim.width &&
         columnXPos - columns.get(columnIdx - 1).width >= 0)
   {
      columnIdx -= 1;
   }

   // Return the index of the last item if that column.
   int itemIdx = ((columnIdx + 1) * itemsInListVer) - 1;
   int itemsCount = fpanel.items.getCount();
   return GMath::Min(itemsCount - 1, itemIdx);
}

void LCmdFilePanelModeBrief::itemsListHasBeenRefreshed ()
{
   layout();
}

void LCmdFilePanelModeBrief::layout ()
{
   GDecoratedWindow::invalidateAll(true);
   GRectangle rect = getWindowRect();
   bool visible = (rect.width > 0 && rect.height > 0);
   if (columns.getCount() == 0 || !hasEverBeenVisible)
   {
      // Always perform the layout code in this method upon first call.
   }
   else
   if (!visible)
   {
      // Panels have newly been toggled off. For speed optimizing reasons,
      // don't recalculate anything in this case. Wait until the panels are
      // toggled back on.
      return;
   }

   // ---
   hasEverBeenVisible = visible;

   // Recalculate all filename items, but only if not already done.
   if (fpanel.items.getCount() > 0 && fpanel.items[0].briefWidth == 0)
      calculateWidthOfAllItemsAfterReloadFilenames();

   // Keep a fresh copy of current window rectangle of the panel
   filesRect = rect;

   // Calculate everything that has to do with column positions, widths,
   // vertical items count, etc.
   calcItems();

   // Be sure to always keep current selected item within visible area
   leftMostColumnXPos = 0;

   int firstVisible = leftMostColumn * itemsInListVer;
   int lastVisible = calcLastVisibleItemNC();
   if (curSelectedItem < firstVisible ||
       curSelectedItem > lastVisible)
   {
      // Current selected item is outside visible area, so make it visible
      int winWidth = filesRect.width;
      int iNewColumn = curSelectedItem / itemsInListVer;

      int columnCount = columns.getCount(); // Make this a fast one
      if (columnCount > 0)
      {
         if (iNewColumn >= 1 && iNewColumn < columnCount)
         {
            int i, xpos;
            for (i = iNewColumn, xpos = winWidth - columns.get(i).width;
                  i > 0 && xpos - columns.get(i-1).width > 0;
                  i--, xpos -= columns.get(i).width);

            leftMostColumn = i;
         }
         else
            leftMostColumn = 0;

         firstVisibleItem = leftMostColumn * itemsInListVer;
      }
   }
}

void LCmdFilePanelModeBrief::drawItem ( int itemIndex )
{
   GRectangle rect;
   GGraphics g(*this);
   if (calcItemRect(itemIndex, rect))
      drawItem(itemIndex, g, rect);
}

void LCmdFilePanelModeBrief::drawItem ( int itemIndex,
                                        GGraphics& g,
                                        const GRectangle& itemRect,
                                        bool isDragOver )
{
   if (itemIndex < 0)
      return;

   LCmdFileItem& fitem = fpanel.items.get(itemIndex);

   if (isDragOver)
   {
      GRectangle r(itemRect);
      if (fitem.isUpDir() || (!fitem.isDirectory() && !fitem.isProgramFileName()))
         r.height = 2; // Draw a line rather than a box.
      g.drawEmphasiseBox(r, GColor::BLACK, 2, true);
      return;
   }

   bool iscurpanel = fpanel.isCurrentPanel();
   bool iscurselected = (itemIndex == curSelectedItem);
   GRectangle rect(itemRect);

   // Paint the thin border of the current selected item.
   if (iscurselected)
   {
      // Draw a thin border around current selected item.
      g.setLineType(GGraphics::LTSolid);
      if (iscurpanel)
         g.setColor(fpanel.colors.itemSelectedThinActive);
      else
         g.setColor(fpanel.colors.itemSelectedThinInactive);
      g.drawRectangle(rect);
      // Don't erase the thin frame just painted.
      rect.inflateRect(-1, -1);
   }

   // ---
   // Find background color.
   GColor bc;
   if (iscurselected)
   {
      if (iscurpanel)
      {
         if (fitem.isMarked())
            bc = fpanel.colors.itemSelectedBckActiveMarked;
         else
            bc = fpanel.colors.itemSelectedBckActive;
      }
      else
      {
         if (fitem.isMarked())
            bc = fpanel.colors.itemBckMarked;
         else
            bc = fpanel.colors.itemSelectedBckInactive;
      }
   }
   else
   {
      if (fitem.isMarked())
         bc = fpanel.colors.itemBckMarked;
      else
         bc = fpanel.colors.itemBck;
   }

   // ---
   // Find foreground (text) color.
   GColor fc;
   if (fitem.isDirectory())
      fc = fpanel.colors.itemDirTxt;
   else
   if (fitem.isZipOrArchiveFile())
      fc = fpanel.colors.itemArchiveTxt;
   else
   if (fitem.isReadOnly())
      fc = fpanel.colors.itemReadOnlyTxt;
   else
   if (fitem.isHidden() || fitem.isSystem())
      fc = fpanel.colors.itemSysHiddenTxt;
   else
      fc = fpanel.colors.itemFileTxt;

   // ---
   // Clear the background area of the item.
   g.drawFilledRectangle(rect, bc);

   // ---
   // Draw the icon and text, using current selected font and colors
   rect.x += 3 - iscurselected;
   GString fullFName = fitem.getFileName();
   if (fpanel.view.showItemIcon)
   {
      int xpos = rect.x;
      int ypos = rect.y + ((itemHeight - iconSize) / 2) - iscurselected;
      fitem.drawTheIcon(g, xpos, ypos, fpanel);
      rect.nudgeX(iconSize + 7);
   }
   else
   {
      rect.x += 3;
   }
   g.drawText(fullFName, rect, fc);
   if (fitem.isHidden() || fitem.isSystem())
   {
      int textw = g.getWidthOfString(fullFName);
      int ypos = rect.y + (rect.height/2) - 1;
      int endx = rect.x + textw;
      g.setColor(fc);
      g.setLineType(GGraphics::LTSolid);
      g.drawLine(rect.x, ypos, endx, ypos);
   }
}

int LCmdFilePanelModeBrief::drawColumn ( int idxColumn )
{
   GGraphics g(*this);
   return drawColumn(g, idxColumn);
}

int LCmdFilePanelModeBrief::drawColumn ( GGraphics& g, int idxColumn )
{
   GRectangle r(filesRect);
   r.x = calcColumnXPos(idxColumn);
   r.width = columns.get(idxColumn).width;
   r.height = itemHeight;
   r.y = filesRect.y + filesRect.height - itemHeight;

   int idxItem = idxColumn * itemsInListVer;
   if (idxItem >= 0)
   {
      // Loop to draw each item in the column
      int lastItem = idxItem + itemsInListVer - 1;
      int itemCount = fpanel.items.getCount();
      while (idxItem <= lastItem && idxItem < itemCount)
      {
         drawItem(idxItem, g, r);
         r.y -= itemHeight;
         idxItem++;
      }
   }

   // Clear the rest of the column area on screen
   r.height = r.y + itemHeight;
   r.y = 0;
   g.drawFilledRectangle(r, fpanel.colors.itemBck);

   return r.x + r.width;
}

int LCmdFilePanelModeBrief::drawColumns ( GGraphics& g, int idxFirstColumn, int count )
{
   int itemIdx;
   int xposNext = 0;
   int idxColumn = idxFirstColumn;
   int winWidth = filesRect.x + filesRect.width - 1;
   int itemCount = fpanel.items.getCount();
   int columnsCount = columns.getCount();
   for (itemIdx = idxFirstColumn * itemsInListVer;
        itemIdx < itemCount && xposNext < winWidth && idxColumn < columnsCount;
        itemIdx += itemsInListVer, idxColumn++)
   {
      xposNext = drawColumn(g, idxColumn);
      if (--count == 0)
         break;
   }

   return xposNext;
}

bool LCmdFilePanelModeBrief::onFontNameSizeChanged ( const GString& fontNameSize )
{
   layout();
   return true;
}

bool LCmdFilePanelModeBrief::onPaint ( GGraphics& g, const class GRectangle& rect )
{
   GRectangle r = getWindowRect();
   if (fpanel.items.isExecuting())
   {
      // The filename reader thread is still working.
      GStringl msg("%Txt_FP_ReadingFilenames"); // Reading filenames...
      g.drawFilledRectangle(r, fpanel.colors.itemBck);
      g.drawText(msg, r, fpanel.colors.itemFileTxt, GGraphics::DUMMY_COLOR, GGraphics::HCENTER);
   }
   else
   {
      r.x = drawColumns(g, leftMostColumn, -1); // Draw all visible columns.
      g.drawFilledRectangle(r, fpanel.colors.itemBck); // Clear the remaining area.
   }

   return true;
}

void LCmdFilePanelModeBrief::scrollHorizontal ( int add )
{
   int absXPos = columns.get(leftMostColumn).xpos - leftMostColumnXPos;
   int prevAbsXPos = absXPos;

   absXPos += add;
   if (absXPos < 0)
      absXPos = 0;
   else
   {
      LCmdFileColumn& clmn = columns.get(columns.getCount() - 1);
      int x = clmn.xpos + clmn.width - (filesRect.x + filesRect.width - 1);
      if (absXPos > x)
         absXPos = x;
   }

   // Find the index of the new leftmost visible column, and the pixel
   // X-position of it with respect to the left edge of the files list window,
   // which is at pixel X-position 0.
   int xPos = 0;
   int columnCount = columns.getCount();
   for (int i=0; i<columnCount; i++)
   {
      LCmdFileColumn& clmn = columns.get(i);
      if (clmn.xpos == absXPos)
      {
         leftMostColumn = i;
         leftMostColumnXPos = 0;
         firstVisibleItem = i * itemsInListVer;
         break;
      }
      else
      if (clmn.xpos > absXPos)
      {
         leftMostColumn = i - 1;
         leftMostColumnXPos = columns.get(i-1).xpos - absXPos;
         firstVisibleItem = (i-1) * itemsInListVer;
         break;
      }
      else
      if (columns.getCount() == 1)
      {
         leftMostColumnXPos = -absXPos;
         break;
      }
      else
      if (i == (columns.getCount() - 1))
      {
         leftMostColumn = i;
         leftMostColumnXPos = xPos - absXPos;
         firstVisibleItem = i * itemsInListVer;
         break;
      }
      else
      {
         xPos += clmn.width;
      }
   }

   // Scroll the content of the files container.
   scrollWindow(prevAbsXPos - absXPos, 0, true, &filesRect);

   updateHScrollBarPos();
}

bool LCmdFilePanelModeBrief::onHScrollLineUp ()
{
   scrollHorizontal(-10);
   return true;
}

bool LCmdFilePanelModeBrief::onHScrollLineDown ()
{
   scrollHorizontal(10);
   return true;
}

bool LCmdFilePanelModeBrief::onHScrollPageUp ()
{
   scrollHorizontal(-(filesRect.x + filesRect.width - 1 - 10));
   return true;
}

bool LCmdFilePanelModeBrief::onHScrollPageDown ()
{
   scrollHorizontal(filesRect.x + filesRect.width - 1 - 10);
   return true;
}

bool LCmdFilePanelModeBrief::onHScrollSliderTrack ( int pos )
{
   int lx = columns.get(leftMostColumn).xpos;
   int absXPos = lx - leftMostColumnXPos;
   scrollHorizontal(-absXPos + pos);
   return true;
}

bool LCmdFilePanelModeBrief::onButton1Up ( int /*xpos*/, int /*ypos*/, const GWindowMessage::InputFlags& /*flags*/ )
{
   if (isMouseCapture())
      captureMouse(false); // Release the mouse capture
   return true;
}

bool LCmdFilePanelModeBrief::onButton1Down ( int xpos, int ypos, const GWindowMessage::InputFlags& flags )
{
   // Activate our frame window (in case LCmd isn't already active).
   setActive();

   // If the user is currently about doing a dynamic filename
   // text search then dismiss it now.
   if (fpanel.headerWin.isPerformingDynamicSearch())
      fpanel.headerWin.dismissDynamicSearch();

   fpanel.activatePanel(); // Activate this file panel.

   int prevSelected = curSelectedItem;
   curSelectedItem = calcItemIdxFromPos(xpos, ypos);
   if (curSelectedItem != prevSelected)
   {
      drawItem(prevSelected);
      drawItem(curSelectedItem);

      // Request the text area of the infobar to show the information
      // about newly selected item.
      fpanel.infoBar.updateAllFileItemInfoCells();

      // Let "everyone" know that the selection has changed.
      fpanel.onItemSelectionHasChanged();
   }

   if (curSelectedItem >= 0 && flags.isControlKeyDown())
      fpanel.toggleTag(curSelectedItem, true);

   captureMouse(true);

   return true;
}

bool LCmdFilePanelModeBrief::onButton1DblClk ( int xpos, int ypos, const GWindowMessage::InputFlags& flags )
{
   if (curSelectedItem >= 0 &&
       curSelectedItem == calcItemIdxFromPos(xpos, ypos))
   {
      if (flags.isAltKeyDown() && flags.isShiftKeyDown())
         fpanel.startSelectedProgram(curSelectedItem);
      else
         fpanel.doEnter(curSelectedItem, true);
   }
   return true;
}

bool LCmdFilePanelModeBrief::onButton2Click ( int xpos, int ypos, const GWindowMessage::InputFlags& /*flags*/ )
{
   // Activate our frame window (in case LCmd isn't already active).
   setActive();

   if (isMouseCapture())
      return true;

   fpanel.activatePanel();
   setVisiblePopupMenu(true, xpos, ypos);

   return true;
}

bool LCmdFilePanelModeBrief::onButton2Down ( int xpos, int ypos, const GWindowMessage::InputFlags& /*flags*/ )
{
   if (isMouseCapture())
      return true;

   // If the user is currently about doing a dynamic filename
   // text search then dismiss it now.
   if (fpanel.headerWin.isPerformingDynamicSearch())
      fpanel.headerWin.dismissDynamicSearch();

   fpanel.activatePanel(); // Specified panel must be active

   int index = calcItemIdxFromPos(xpos, ypos);
   if (index == curSelectedItem)
      return true;

   int prevSelected = curSelectedItem;
   curSelectedItem = index;

   if (fpanel.markedFilesCount > 0)
   {
      if (index < 0 || !fpanel.items.get(index).isMarked())
      {
         // User has right-clicked on an untagged item even though there 
         // are other items currently being tagged. In order to prevent
         // confusion, untag all filename items so that if the user 
         // selects a command from the popup window the command will 
         // work only on the single selected (right clicked) item.
         // This is the same as in the Windows Explorer, which most 
         // users probably expect.
         fpanel.unselectAll();
      }
   }

   drawItem(prevSelected);
   drawItem(curSelectedItem);

   // Request the text area of the infobar to show the information
   // about newly selected item.
   fpanel.infoBar.updateAllFileItemInfoCells();

   // Let "everyone" know that the selection has changed.
   fpanel.onItemSelectionHasChanged();

   return true;
}

bool LCmdFilePanelModeBrief::onMouseMove ( int xpos, int ypos, const GWindowMessage::InputFlags& flags )
{
   if (!isMouseCapture())
      return false;

   if (curSelectedItem >= 0)
   {
      int index = calcItemIdxFromPos(xpos, ypos);
      if (index >= 0)
      {
         LCmdFileItem& fitem = fpanel.items.get(curSelectedItem);
         bool prevIsTagged = fitem.isMarked();

         // Activate and repaint the calculated item
         int prevSelected = curSelectedItem;
         if (index != prevSelected)
         {
            curSelectedItem = index;
            drawItem(prevSelected);
            drawItem(index);

            // Request the text area of the infobar to show the information
            // about newly selected item
            fpanel.infoBar.updateAllFileItemInfoCells();

            // Let "everyone" know that the selection has changed.
            fpanel.onItemSelectionHasChanged();
         }

         if (flags.isControlKeyDown())
         {
            LCmdFileItem& itm = fpanel.items.get(index);
            if (itm.isMarked() != prevIsTagged)
               fpanel.toggleTag(index, true);
         }
      }
   }

   return false; // Return true if and only if we have actually set the mouse cursor.
}

bool LCmdFilePanelModeBrief::onInitMenu ()
{
   lcmd->mainWin.updateCommandStates();
   return true;
}

int LCmdFilePanelModeBrief::navigateDown ()
{
   int scrollWidth;
   int prevIndex = curSelectedItem;
   int newIndex = prevIndex + 1;
   if (newIndex >= fpanel.items.getCount())
      return -1;

   curSelectedItem = newIndex;

   // Unselect previously selected item
   drawItem(prevIndex);

   // Calculate index of column to which newly selected item belongs
   int newColumn = newIndex / itemsInListVer;
   int lastVisibleNC = calcLastVisibleItemNC();
   GRectangle rScroll(filesRect);

   // -----------------------------------------------------------------------
   // If requested item is in leftmost column, which may be clipped.
   // (This can happen if user has selected one of those clipped items by
   // using the mouse.)

   if (newColumn == leftMostColumn &&
       leftMostColumnXPos < 0)
   {
      scrollWidth = -leftMostColumnXPos; // Convert to absolute value
      leftMostColumnXPos = 0;
   }

   // -----------------------------------------------------------------------
   // If requested item is already within visible area

   else
   if (newIndex >= firstVisibleItem &&
       newIndex <= lastVisibleNC)
   {
      drawItem(newIndex);
      return newIndex;
   }

   // -----------------------------------------------------------------------
   // If requested item is in rightmost column, which may be clipped.
   // (This can happen if user has selected one of those clipped items by
   // using the mouse.)

   else
   if (newIndex > lastVisibleNC)
   {
      int winWidth = rScroll.width;
      int xpos = calcColumnXPos(newColumn);
      int columnWidth = columns.get(newColumn).width;
      bool visible = xpos > winWidth ? false : true;

      scrollWidth = winWidth - xpos - columnWidth;

      int i = newColumn;
      for (xpos = winWidth - columnWidth;
           xpos > 0 && i >= 0;
           xpos -= columns.get(--i).width);

      leftMostColumnXPos = (columnWidth < winWidth) ? xpos : 0;
      leftMostColumn = (i > 0) ? i : 0;
      firstVisibleItem = leftMostColumn * itemsInListVer;

      if (!visible)
      {
         // Previous selected item was completely outside visible area, so
         // request whole file list window to be redrawn (just in case).
         GDecoratedWindow::invalidateAll(true);
         updateHScrollBarPos();
         return newIndex;
      }
   }

   // --------------------------------------------------------------------------
   // The requested column is completely outside visible area, so activate
   // it as when random selection

   else
   if (newColumn < leftMostColumn)
   {
      return navigateRandom(newIndex);
   }

   // --------------------------------------------------------------------------
   // If requested item is in the column next to the right of currently
   // rightmost visible column, then perform a normal right scrolling to
   // activate the next column to the right

   else
   {
      leftMostColumn += 1;
      firstVisibleItem = leftMostColumn * itemsInListVer;
      scrollWidth = columns.get(leftMostColumn).width;
   }

   // Perform the scrolling
   scrollWindow(scrollWidth, 0, true, &rScroll);

   // Redraw the newly activated column
   drawColumn(newColumn);

   // ---
   updateHScrollBarPos();

   return newIndex;
}

int LCmdFilePanelModeBrief::navigateUp ()
{
   int i;
   int scrollWidth;
   int lastVisibleNC;
   int newColumn;
   int prevIndex = curSelectedItem;
   int newIndex = prevIndex - 1;
   if (newIndex < 0)
      return -1;

   curSelectedItem = newIndex;

   // Unselect previously active item
   drawItem(prevIndex);

   // Calculate index of column to which newly selected item belongs
   newColumn = newIndex / itemsInListVer;
   lastVisibleNC = calcLastVisibleItemNC();
   GRectangle rScroll(filesRect);

   // -----------------------------------------------------------------------
   // If requested item is in leftmost column, which may be clipped.
   // (This can happen if user has selected one of those clipped items by
   // using the mouse.)

   if (newColumn == leftMostColumn &&
       leftMostColumnXPos < 0)
   {
      scrollWidth = -leftMostColumnXPos; // Convert to absolute value
      leftMostColumnXPos = 0;
   }

   // -----------------------------------------------------------------------
   // If requested item is already within visible area

   else
   if (newIndex >= firstVisibleItem &&
       newIndex <= lastVisibleNC)
   {
      drawItem(newIndex);
      return newIndex;
   }

   // -----------------------------------------------------------------------
   // If requested item is in rightmost column, which may be clipped.
   // (This can happen if user has selected one of those clipped items by
   // using the mouse.)

   else
   if (newIndex > lastVisibleNC)
   {
      int winWidth = rScroll.width;
      int xpos = calcColumnXPos(newColumn);
      int columnWidth = columns.get(newColumn).width;

      scrollWidth = winWidth - xpos - columnWidth;

      for (xpos = winWidth - columnWidth, i = newColumn;
           xpos > 0 && i >= 0;
           xpos -= columns.get(--i).width);

      leftMostColumnXPos = xpos;
      leftMostColumn = (i > 0) ? i : 0;
      firstVisibleItem = leftMostColumn * itemsInListVer;
   }

   // --------------------------------------------------------------------------
   // If requested item is in the column next to the left of currently
   // leftmost visible column, then perform a normal left scrolling to activate
   // the next column to the left

   else
   if (newColumn == leftMostColumn - 1)
   {
      leftMostColumn -= 1;
      firstVisibleItem = leftMostColumn * itemsInListVer;
      scrollWidth = columns.get(leftMostColumn).width;
   }

   // --------------------------------------------------------------------------
   // The requested column is completely outside visible area, so activate
   // it as when random selection

   else
   {
      return navigateRandom(newIndex);
   }

   // Perform the scrolling
   scrollWindow(scrollWidth, 0, true, &rScroll);

   // Redraw the newly activated column
   drawColumn(newColumn);

   // ---
   updateHScrollBarPos();

   return newIndex;
}

int LCmdFilePanelModeBrief::navigateEnd ()
{
   int prevIndex = curSelectedItem;
   int newIndex = fpanel.items.getCount() - 1;
   if (newIndex == prevIndex)
      return newIndex;

   curSelectedItem = newIndex;

   // Unselect previously selected item
   drawItem(prevIndex);

   // --------------------------------------------------------------------------
   // If requested item is already within visible area

   if (newIndex >= firstVisibleItem &&
       newIndex <= calcLastVisibleItemNC())
   {
      drawItem(newIndex);
      return newIndex;
   }

   // --------------------------------------------------------------------------

   int i;
   int newColumn = newIndex / itemsInListVer;
   int winWidth = filesRect.width;

   // If the new column is to wide to show in window, then just activate and
   // repaint it before return
   if (columns.get(newColumn).width >= winWidth)
   {
      leftMostColumnXPos = 0;
      leftMostColumn = newColumn;
      firstVisibleItem = leftMostColumn * itemsInListVer;
      GDecoratedWindow::invalidateAll(true);
      updateHScrollBarPos();
      return newIndex;
   }

   // Find index and position of new leftmost visible column
   i = newColumn;
   leftMostColumnXPos = winWidth;
   do
   {
      leftMostColumnXPos -= columns.get(i).width;
      if (leftMostColumnXPos < 0)
         break;
   }
   while (--i >= 0);

   leftMostColumn = (i >= 0) ? i : 0;
   firstVisibleItem = leftMostColumn * itemsInListVer;

   drawItem(newIndex);
   GDecoratedWindow::invalidateAll(true);
   updateHScrollBarPos();
   return newIndex;
}

int LCmdFilePanelModeBrief::navigateHome ()
{
   if (curSelectedItem == 0)
      return 0;

   int prevIndex = curSelectedItem;
   int iPrev1Visible = firstVisibleItem;
   int iPrevLeftMostColumnXPos = leftMostColumnXPos;

   leftMostColumn = 0;
   leftMostColumnXPos = 0;
   curSelectedItem = 0;
   firstVisibleItem = 0;

   if (iPrev1Visible == 0 &&
       iPrevLeftMostColumnXPos == 0)
   {
      // Very first item is already at upper left corner
      drawItem(prevIndex);
      drawItem(0);
   }
   else
   {
      GDecoratedWindow::invalidateAll(true);
   }

   updateHScrollBarPos();
   return 0;
}

int LCmdFilePanelModeBrief::navigateLeft ()
{
   int scrollWidth;
   int newColumn;

   int prevIndex = curSelectedItem;
   int newIndex = prevIndex - itemsInListVer;
   if (newIndex < 0)
      newIndex = 0;

   if (newIndex == prevIndex)
      return newIndex;

   curSelectedItem = newIndex;

   // Unselect previously selected item
   drawItem(prevIndex);

   // -----------------------------------------------------------------------
   // If requested item is in current leftmost visible column, and that
   // column is currently clipped (part of it is outside visible screen
   // area)

   newColumn = newIndex / itemsInListVer;

   if (newColumn == leftMostColumn &&
       leftMostColumnXPos < 0)
   {
      scrollWidth = -leftMostColumnXPos; // Convert to absolute
      leftMostColumnXPos = 0;
   }

   // -----------------------------------------------------------------------
   // If requested item is already within visible area

   else
   if (newIndex >= firstVisibleItem &&
       newIndex <= calcLastVisibleItemNC())
   {
      drawItem(newIndex);
      return newIndex;
   }

   // --------------------------------------------------------------------------
   // Activate the left neightbor column of previos leftmost column

   else
   if (newColumn == leftMostColumn - 1)
   {
      leftMostColumn -= 1;
      leftMostColumnXPos = 0;
      firstVisibleItem = leftMostColumn * itemsInListVer;
      scrollWidth = columns.get(leftMostColumn).width;
   }

   // --------------------------------------------------------------------------
   // Previous item was completeley outside visible area, so activate
   // it as when random selection

   else
   {
      return navigateRandom(newIndex);
   }

   // Perform the scrolling
   GRectangle rScroll(filesRect);
   scrollWindow(scrollWidth, 0, true, &rScroll);

   // Redraw the new leftmost column
   drawColumn(leftMostColumn);

   updateHScrollBarPos();
   return newIndex;
}

int LCmdFilePanelModeBrief::navigateRight ()
{
   int prevIndex = curSelectedItem;
   int newIndex = prevIndex + itemsInListVer;
   if (newIndex >= fpanel.items.getCount())
      newIndex = fpanel.items.getCount() - 1;

   if (newIndex == prevIndex)
      return newIndex;

   curSelectedItem = newIndex;

   // Unselect previously selected item
   drawItem(prevIndex);

   // --------------------------------------------------------------------------
   // If requested item is already within visible area

   if (newIndex >= firstVisibleItem &&
       newIndex <= calcLastVisibleItemNC())
   {
      drawItem(newIndex);
      return newIndex;
   }

   // --------------------------------------------------------------------------
   // Activate the right neightbor column of previos rightmost column

   GRectangle rScroll(filesRect);
   int winWidth = rScroll.width;

   // Calculate index of column to which newly selected item belongs
   int newColumn = newIndex / itemsInListVer;

   // Calculate width of area to scroll
   int i;
   int scrWidth = leftMostColumnXPos;
   int nrOfColumns = columns.getCount();
   for (i = leftMostColumn;
        i < nrOfColumns;
        i++)
   {
      scrWidth += columns.get(i).width;
      if (scrWidth > winWidth)
      {
         if (i == leftMostColumn)
         {
            // Previous active column was to wide to show in window, so just
            // force the new column to be visible and repaint before return
            int winWidth = rScroll.width;
            int columnWidth = columns.get(newColumn).width;

            int xpos;
            for (xpos = winWidth - columnWidth, i = newColumn;
                 xpos > 0 && i >= 0;
                 xpos -= columns.get(--i).width);

            leftMostColumnXPos = (columnWidth < winWidth) ? xpos : 0;
            leftMostColumn = (i > 0) ? i : 0;
            firstVisibleItem = leftMostColumn * itemsInListVer;

            GDecoratedWindow::invalidateAll(true);
            return newIndex;
         }

         scrWidth -= winWidth;
         break;
      }
   }

   // If the new column is to wide to show in window, then just activate and
   // repaint it before return
   if (columns.get(newColumn).width >= winWidth ||
       calcColumnXPos(newColumn) > calcColumnXPos(i) ||
       newColumn < leftMostColumn)
   {
      leftMostColumnXPos = 0;
      leftMostColumn = newColumn;
      firstVisibleItem = leftMostColumn * itemsInListVer;
      GDecoratedWindow::invalidateAll(true);
      updateHScrollBarPos();
      return newIndex;
   }

   // Perform the scrolling
   scrollWindow(-scrWidth, 0, true, &rScroll);

   // Find index and position of new leftmost visible column
   leftMostColumnXPos = winWidth;
   do
   {
      leftMostColumnXPos -= columns.get(i).width;
      if (leftMostColumnXPos < 0)
      {
         leftMostColumn = i;
         break;
      }
   }
   while (--i >= 0);

   firstVisibleItem = leftMostColumn * itemsInListVer;

   // Redraw the newly activated column
   drawColumn(newColumn);

   updateHScrollBarPos();
   return newIndex;
}

int LCmdFilePanelModeBrief::navigatePageDown ()
{
   return -1;
}

int LCmdFilePanelModeBrief::navigatePageUp ()
{
   return -1;
}

int LCmdFilePanelModeBrief::navigateRandom ( int index )
{
   if (index < 0)
      index = 0;

   int count = fpanel.items.getCount();
   if (count <= 0)
      return -1;

   if (index >= count)
      index = count - 1;

   int prevIndex = curSelectedItem;
   int lastVisibleNC = calcLastVisibleItemNC();

   // Test if the requested item is completely visible, without being
   // clipped at all.
   bool visible = leftMostColumnXPos == 0 && // Is clipped if != 0
                  index >= firstVisibleItem &&
                  index <= lastVisibleNC;

   if (visible && index == prevIndex)
   {
      // Redraw the item, in case marking of it has been toggled on/off.
      drawItem(index);
      return index;
   }

   curSelectedItem = index;

   // Unselect previously selected item
   drawItem(prevIndex);

   // --------------------------------------------------------------------------
   // If requested item is already within none-clipped visible area

   if (visible)
   {
      drawItem(index);
      return index;
   }

   // --------------------------------------------------------------------------
   // Make requested item visible

   int winWidth = filesRect.width;
   int newColumn = index / itemsInListVer;

   int i = newColumn;
   for (int xpos = winWidth - columns.get(i).width;
        i > 0 && xpos - columns.get(i-1).width > 0;
        xpos -= columns.get(--i).width);

   leftMostColumn = i;
   leftMostColumnXPos = 0;
   firstVisibleItem = leftMostColumn * itemsInListVer;

   // Repaint whole list of filenames in fpanel
   GDecoratedWindow::invalidateAll(true);

   updateHScrollBarPos();
   return index;
}

void LCmdFilePanelModeBrief::calculateWidthOfAllItemsAfterReloadFilenames ()
{
   GGraphics g(this);
   LCmdOptions& opt = LCmdOptions::GetOptions();
   int addExtra = opt.various.addToItemWidth + 2;
   int addIconWidth = (fpanel.view.showItemIcon ? fpanel.iconSize + 2 : 0);
   for (int i=0, num=fpanel.items.getCount(); i<num; i++)
   {
      LCmdFileItem& fitem = fpanel.items.get(i);
      GString fname = fitem.path.getFileName();
      int strw = g.getWidthOfString(fname);
      int w = strw + addIconWidth + addExtra;
      if (w < 6)
         w = 6;
      fitem.briefWidth = w;
   }
}
