/* --------------------------------------------------------------------------
 *
 * Copyright (C) 2007 Leif Erik Larsen, Kjerringvik, Norway.
 *
 * This file is part of the Open Source Edition of Larsen Commander, as
 * available from http://home.online.no/~leifel/lcmd/.  This code is free 
 * software; you can redistribute it and/or modify it under the terms of 
 * the GNU General Public License version 3 only, as published by the 
 * Free Software Foundation.  
 *
 * This code is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 * version 3 at http://www.gnu.org/licenses/gpl-3.0.txt for more details 
 * (a copy is included in the LICENSE file that accompanied this code).
 *
 * ------------------------------------------------------------------------ */

#include "lcmd/LCmdFilePanelModeTree.h"
#include "lcmd/LCmdFilePanel.h"
#include "lcmd/LCmdFilePanelInfoBar.h"
#include "lcmd/LCmdFilePanelFrame.h"

#include "glib/gui/layout/GBorderLayout.h"
#include "glib/gui/tree/GTreePath.h"
#include "glib/util/GTokenizer.h"
#include "glib/util/GDriveInfo.h"
#include "glib/vfs/GVfsLocal.h"
#include "glib/io/GIOException.h"
#include "glib/GProgram.h"

/**
 * An instance of this class is used as the "user data" of each and every 
 * node in our tree model. The point is to keep track of the Virtual File 
 * System that each node belongs to, since we can browse directly inside 
 * archive files as well as system drives.
 *
 * @author  Leif Erik Larsen
 * @since   2006.02.15
 */
class MyNode : public GObject
{

public:

   /** The Virtual File System of where this node originates. */
   GVfs* vfs;
   bool autoDeleteVfs;

   /** If this is a drive node then this variable will be null. */
   LCmdFileItem* fitem;
   bool autoDeleteFItem;

   /** If this is a "Drive Node" then this variable will contain the name of the drive, e.g. "C:". */
   GString driveName;

   /** Create a node representing a directory or filename item. */
   MyNode ( GVfs* vfs, bool autoDeleteVfs, LCmdFileItem* fitem, bool autoDeleteFItem )
      :vfs(vfs),
       autoDeleteVfs(autoDeleteVfs),
       fitem(fitem),
       autoDeleteFItem(autoDeleteFItem)
   {
      if (fitem == null)
         gthrow_(GIllegalArgumentException("fitem==null"));
   }

   /** Create a node representing a root drive. Used on Windows and OS/2 only. */
   MyNode ( GVfs* vfs, bool autoDeleteVfs, const GString& driveName )
      :vfs(vfs),
       autoDeleteVfs(autoDeleteVfs),
       fitem(null),
       autoDeleteFItem(false),
       driveName(driveName)
   {
      if (driveName == "")
         gthrow_(GIllegalArgumentException("driveName==\"\""));
   }

   virtual ~MyNode ()
   {
      if (autoDeleteFItem)
         delete fitem;
      if (autoDeleteVfs)
         delete vfs;
   }

private:

   /** Disable the copy-constructor. */
   MyNode ( const MyNode& src ) {}

   /** Disable the assignment operator. */
   MyNode& operator= ( const MyNode& src ) { return *this; }

public:

   /**
    * Return true if and only if this node represent a drive, 
    * and not a directory.
    *
    * @author  Leif Erik Larsen
    * @since   2006.02.15
    */
   bool isDriveNode () const
   {
      return driveName != "";
   }

   /**
    * This method is called to provide the text to represent the node in
    * the Tree View. We return the name of the file item (incl extension).
    *
    * @author  Leif Erik Larsen
    * @since   2006.02.15
    */
   virtual GString toString () const
   {
      if (isDriveNode())
         return driveName;
      else
         return fitem->getFileName();
   }
};

LCmdFilePanelModeTree::LCmdFilePanelModeTree ( LCmdFilePanel& fpanel,
                                               const GString& winName )
                      :GWindow(&fpanel.tabs, winName, GString::Empty, WS2_OS2Y),
                       LCmdFilePanelModeAbstract(fpanel),
                       fpanel(fpanel),
                       rootNode(true, new GString("Root"), true),
                       dirTree(&rootNode, false),
                       treeView("View", // name
                                GBorderLayout::CENTER, // constraints
                                *this, // parentWin
                                false, // border
                                true, // hasLines
                                false, // linesAtRoot
                                true, // hasButtons
                                false, // showRoot
                                false, // grabFocus
                                WS_VISIBLE, // winStyle
                                0) // winStyle2
{   
   setLayoutManager(new GBorderLayout(), true);
   treeView.setTreeModel(&dirTree, false);
   treeView.addTreeViewListener(this);
   treeView.setBackgroundColor(fpanel.colors.itemBck);
   treeView.setForegroundColor(fpanel.colors.itemFileTxt);

   // Activate the popup menu of which to display when user
   // right-click on a file item or on some other space of this window.
   GProgram& prg = GProgram::GetProgram();
   setPopupMenu("FileItemContextMenu", prg.isUseFancyPopupMenues());
}

LCmdFilePanelModeTree::~LCmdFilePanelModeTree ()
{
   treeView.removeTreeViewListener(this);
}

void LCmdFilePanelModeTree::invalidateAll ( bool inclChildren ) const
{
   GWindow::invalidateAll(inclChildren);
   treeView.invalidateAll(inclChildren);
}

void LCmdFilePanelModeTree::invalidateRect ( const GRectangle& rect ) const
{
   GWindow::invalidateRect(rect);
   treeView.invalidateRect(rect);
}

void LCmdFilePanelModeTree::onViewHasBeenActivated ()
{
   layout();
   fpanel.frameWin.layout();
   fpanel.frameWin.headerWin.layout();
}

bool LCmdFilePanelModeTree::isHorizontallyScrollable () const
{
   // TODO: ???
   return true;
}

void LCmdFilePanelModeTree::itemsListHasBeenRefreshed ()
{
   refreshContent();
}

void LCmdFilePanelModeTree::layout ()
{
   GWindow::layout();
}

int LCmdFilePanelModeTree::calcItemIdxFromPos ( int xpos, int ypos ) const
{
   // TODO: ???
   return -1;
}

bool LCmdFilePanelModeTree::calcItemRect ( int itemIndex, GRectangle& rect ) const
{
   // TODO: ???
   rect.clear();
   return false;
}

LCmdFileItem* LCmdFilePanelModeTree::getCurrentSelectedFileItem ()
{
   GTreeNode* curNode = treeView.getSelectedNode();
   if (curNode == null)
      return null; // No current selection.
   MyNode* userObj = dynamic_cast<MyNode*>(curNode->getUserObject());
   if (userObj == null) // True only for the "Root" node (might be visible during testing).
      return null;
   LCmdFileItem* item = userObj->fitem;
   if (item == null)
      return null; // Current selection is a drive node.
   return item;
}

int LCmdFilePanelModeTree::getCurrentSelectedIndex () const
{
   // TODO: ???
   return -1;
}

int LCmdFilePanelModeTree::getFirstVisibleIndex () const
{
   // TODO: ???
   return -1;
}

void LCmdFilePanelModeTree::drawItem ( int /*itemIndex*/ )
{
   // TODO: ???
}

void LCmdFilePanelModeTree::drawItem ( int /*itemIndex*/, class GGraphics& /*g*/, const class GRectangle& /*itemRect*/, bool /*isDragOver*/ )
{
   // TODO: ???
}

bool LCmdFilePanelModeTree::onPaint ( GGraphics& g, const GRectangle& rect )
{
   // TODO: ???
   g.drawFilledRectangle(rect, fpanel.colors.itemBck);
   return true;
}

int LCmdFilePanelModeTree::navigateDown ()
{
   // TODO: This code should probably be part of some method for default key-down handling inside low-level class GTreeView.
   GMutableTreeNode* cur = treeView.getSelectedNode();
   if (cur == null)
      return -1;
   GMutableTreeNode* next = cur->getNextNode();
   if (next == null || (next->isRoot() && !treeView.isShowRoot()))
      return -1;
   treeView.setSelectedNode(*next);
   return -1; // TODO: Should return new selection index, but has the Win32 TreeView this?
}

int LCmdFilePanelModeTree::navigateUp ()
{
   // TODO: This code should probably be part of some method for default key-down handling inside low-level class GTreeView.
   GMutableTreeNode* cur = treeView.getSelectedNode();
   if (cur == null)
      return -1;
   GMutableTreeNode* next = cur->getPreviousNode();
   if (next == null || (next->isRoot() && !treeView.isShowRoot()))
      return -1;
   treeView.setSelectedNode(*next);
   return -1; // TODO: Should return new selection index, but has the Win32 TreeView this?
}

int LCmdFilePanelModeTree::navigateEnd ()
{
   // TODO: ???
   return -1;
}

int LCmdFilePanelModeTree::navigateHome ()
{
   // TODO: ???
   return -1;
}

int LCmdFilePanelModeTree::navigateLeft ()
{
   // TODO: ???
   return -1;
}

int LCmdFilePanelModeTree::navigateRight ()
{
   // TODO: ???
   return -1;
}

int LCmdFilePanelModeTree::navigatePageDown ()
{
   // TODO: ???
   return -1;
}

int LCmdFilePanelModeTree::navigatePageUp ()
{
   // TODO: ???
   return -1;
}

int LCmdFilePanelModeTree::navigateRandom ( int index )
{
   // TODO: ???
   return -1;
}

void LCmdFilePanelModeTree::treeNodeSelectionHasChanged ()
{
   GMutableTreeNode* node = treeView.getSelectedNode();
   if (node == null)
      return;

   // Request the text area of the infobar to show the information
   // about newly selected item.
   fpanel.infoBar.updateAllFileItemInfoCells();

   // Get the File System path of the newly selected node.
   GString fsPath(256);
   GTreePath nodePath = node->getPath();
   for (int i=0, num=nodePath.getPathCount(); i<num; i++)
   {
      GTreeNode& next = nodePath.getPathComponent(i);
      GString nodeStr = next.toString();
      MyNode* userObj = dynamic_cast<MyNode*>(next.getUserObject());
      if (userObj == null) // True only for the "Root" node (might be visible during testing).
         continue;
      fsPath += nodeStr;
      GVfs* vfs = userObj->vfs;
      if (vfs != null) // Should always be true, but in case.
         vfs->slash(fsPath);
   }

   // Activate the newly selected file system path of the tree-view.
   bool temp = true; // TODO: Don't reload filename items.
   fpanel.walkDir(fsPath, temp);

   // Activate the same directory in the opposite file panel.
   const GString timerID = "ActivateNodeInOppositePanel";
   if (isTimerStarted(timerID))
      resetTimer(timerID);
   else
      startTimer(timerID, 500); // TODO: Delay should be user customizable. 
}

bool LCmdFilePanelModeTree::onTimer ( const GString& timerID, GObject* userData )
{
   if (timerID == "ActivateNodeInOppositePanel")
   {
      stopTimer(timerID);
      GString path = fpanel.getCurrentVfsDirectory(false);
      LCmdFilePanel& ofp = fpanel.getOppositePanel();
      ofp.walkDir(path);
      return true;
   }
   else
   {
      return GWindow::onTimer(timerID, userData);
   }
}

void LCmdFilePanelModeTree::treeNodeIsCollapsing ( GTreeNode& node )
{
   // The specified node is about to collapse.
   // We have nothing to do here, since we want to keep the children 
   // in memory until the Tree is explicitly refreshed with e.g. Ctrl+R.
}

void LCmdFilePanelModeTree::treeNodeIsExpanding ( GTreeNode& node )
{
   // The specified node is about to expand.
   // Dynamically load children nodes at this point, if the specified 
   // node isn't already physically loaded with its children.
   if (node.getChildCount() > 0)
      return; // This node has already loaded its children.

   // In order for us to dynamically load children into the node, 
   // the node need to be mutable. This is probably always the case,
   // but test it anyway to be sure.
   GMutableTreeNode* mutableNode = dynamic_cast<GMutableTreeNode*>(&node);
   if (mutableNode == null)
      return; // Not a mutable node.

   MyNode* userObj = dynamic_cast<MyNode*>(node.getUserObject());
   if (userObj == null) 
      return; // Not a "MyNode".

   // Respect nested- VFSs.
   GString dirWithinVfs;
   if (userObj->isDriveNode())
   {
      dirWithinVfs = userObj->driveName; // Root directory inside the Local File System.
   }
   else
   if (userObj->fitem->isDirectory())
   {
      // A directory on the same file system as of the node it self.
      dirWithinVfs = userObj->fitem->getFullPath();
   }
   else
   if (userObj->fitem->isZipOrArchiveFile())
   {
      // An archive file. Open the archive file as a Virtual File System.
      GTreeNode* parentNode = node.getParent();
      if (parentNode == null) // Should never happen, but in case.
         return;
      MyNode* parentUserObj = dynamic_cast<MyNode*>(parentNode->getUserObject());
      if (parentUserObj == null) 
         return; // Not a "MyNode".
      try {
         GVfs& parentVfs = *parentUserObj->vfs;
         GString path = userObj->fitem->getFullPath();
         GVfsLocal& localVfs = fpanel.vfs.root();
         GVfs* vfs = LCmdFilePanel::CreateVfs(parentVfs, path, localVfs);
         userObj->vfs = vfs;
         userObj->autoDeleteVfs = true;
      } catch (GIOException& /*e*/) {
         return;
      }
      dirWithinVfs = ""; // Root directory inside the archive file.
   }
   else
   {
      // Not a directory, and not an archive file.
      return;
   }

   // ---
   GVfs& fs = *userObj->vfs;
   GFileItem fitem;
   fs.slash(dirWithinVfs);

   int countAdded = 0;
   int hdir = fs.findFirst(fitem, dirWithinVfs);
   if (hdir != 0) do 
   {
      if (fitem.isThisDir() || fitem.isUpDir())
         continue;
      bool isDir = fitem.isDirectory();
      bool isSubVfs = !isDir && fitem.isZipOrArchiveFile();
      if (!isDir && !isSubVfs)
         continue;

      GString fname = fitem.getFileName();
      GString fitemPath = dirWithinVfs + fname;
      LCmdFileItem* fitem = new LCmdFileItem(fs, fitemPath);
      MyNode* userData = new MyNode(&fs, false, fitem, true);

      bool allowsChildren = isSubVfs || !fs.isDirectoryEmpty(fitemPath, GVfs::IDE_IgnoreFiles);
      GMutableTreeNode* child = new GMutableTreeNode(allowsChildren, userData, true);
      child->setIconClosed(isSubVfs ? "IDP_DIRZIP" : "FolderClosed"); // TODO: ???
      child->setIconOpened(isSubVfs ? "IDP_DIRZIP" : "FolderOpened"); // TODO: ???
      dirTree.insertNodeInto(child, *mutableNode, countAdded, true);
      countAdded += 1;
   } 
   while (fs.findNext(hdir, fitem));
   if (hdir != 0)
      fs.findClose(hdir);

   // If we did not find any children nodes at all we are better visualizing 
   // this to the user by making the +/- button next to the node text 
   // to disappear. This can happen on archive file nodes only, since we 
   // don't test if they have any contained sub-directories until the users
   // first attempt to expand it.
   if (countAdded == 0)
      mutableNode->setAllowsChildren(false);
}

void LCmdFilePanelModeTree::refreshContent ()
{
   // Remove all children of the root.
   for (int i=rootNode.getChildCount()-1; i>=0; i--)
   {
      GTreeNode& node = rootNode.getChildAt(i);
      dirTree.removeNodeFromParent(node);
   }

   // Add the nested levels of directory nodes, from the root of the local 
   // file system and down to the current directory inside the current 
   // virtual file system.
   GMutableTreeNode* parent = &rootNode;
   for (int i=0, count=fpanel.vfs.getCount(); i<count; i++)
   {
      GVfs& fs = fpanel.vfs.get(i);
      
      // Split the current VS directory into its separate elements, using 
      // a tokenizer with the directory slash character as its delimiter.
      GString vfdir = fs.getCurrentDirectory(false);
      GVector<GString> vfdirElms(16);
      for (GTokenizer tknzr(vfdir, fs.getSlashStr(), GString::Empty, true); ; ) 
      {
         const GToken* tok = tknzr.getNextToken();
         if (tok->isEmpty())
            break; // No more tokens (directory elements).
         GString tokStr = tok->toString();
         vfdirElms.add(tokStr);
      }

      // ---
      bool isDriveNode = false;
      GString vfSelfName = fs.getLogicalSelfName();
      if (vfSelfName == "")
      {
         // This is the local file system.
         // Add a node for each available drive.
         GString nextNodeName = vfdirElms.elementAt(0);
         nextNodeName.toUpperCase(); // Make sure drive name X: is uppercase.
         GMutableTreeNode* parentOfDriveNodes = parent;
         int indexOfNextDriveNode = 0;
         for (char driveLetter='A'; driveLetter<='Z'; driveLetter++)
         {
            if (!GDriveInfo::IsAValidDriveLetter(driveLetter))
               continue;

            GString driveName("%s:", GVArgs(driveLetter));
            MyNode* userData = new MyNode(&fs, false, driveName);

            GMutableTreeNode* driveNode = new GMutableTreeNode(true, userData, true);
            driveNode->setIconClosed("IDP_DRIVE_HARDDISK"); // TODO: ???
            driveNode->setIconOpened("IDP_DRIVE_HARDDISK"); // TODO: ???
            dirTree.insertNodeInto(driveNode, *parentOfDriveNodes, indexOfNextDriveNode++, true);
            if (nextNodeName == driveName)
            {
               isDriveNode = true;
               parent = driveNode;
            }
         }
      }
      else
      {
         // This is a Virtual File System (e.g. a ZIP-file).
         // Add the name of the "ZIP-file" it self, as a directory node.
         GVfsLocal& localVfs = fpanel.vfs.root();
         GString archiveFilePath = fs.getPhysicalSelfName();
         LCmdFileItem* fitem = new LCmdFileItem(localVfs, archiveFilePath);
         MyNode* userData = new MyNode(&fs, false, fitem, true);
         GMutableTreeNode* vfSelfNode = new GMutableTreeNode(true, userData, true);
         vfSelfNode->setIconClosed("IDP_DIRZIP"); // TODO: ???
         vfSelfNode->setIconOpened("IDP_DIRZIP"); // TODO: ???
         dirTree.insertNodeInto(vfSelfNode, *parent, 0, true);
         parent = vfSelfNode;
      }

      // ---
      GFileItem fitem;
      GString path(128);
      GMutableTreeNode* nextChild = null;
      bool ignoreCase = !fs.isFileNameCaseSensitive();
      int dirElmCount = vfdirElms.getCount();
      int dirElmIdx = 0;
      if (isDriveNode)
      {
         path = vfdirElms.elementAt(0);
         fs.slash(path);
         dirElmIdx = 1; // The drive node is already inserted into the tree view.
      }
      else
      {
         // An empty string to findFirst() means "current directory".
         // Therefore, we must tell findFirst() to look in the root 
         // directory of the Virtual File System.
         path = fs.getSlashStr();
      }
      for (; dirElmIdx<dirElmCount; dirElmIdx++)
      {
         const GString& nodeName = vfdirElms.elementAt(dirElmIdx);
         int countAdded = 0;
         int hdir = fs.findFirst(fitem, path);
         if (hdir != 0) do 
         {
            if (fitem.isThisDir() || fitem.isUpDir())
               continue;
            bool isDir = fitem.isDirectory();
            bool isSubVfs = !isDir && fitem.isZipOrArchiveFile();
            if (!isDir && !isSubVfs)
               continue;

            GString fname = fitem.getFileName();
            GString fitemPath = path + fname;
            LCmdFileItem* fitem = new LCmdFileItem(fs, fitemPath);
            MyNode* userData = new MyNode(&fs, false, fitem, true);

            bool allowsChildren = isSubVfs || !fs.isDirectoryEmpty(fitemPath, GVfs::IDE_IgnoreFiles);
            GMutableTreeNode* child = new GMutableTreeNode(allowsChildren, userData, true);
            child->setIconClosed(isSubVfs ? "IDP_DIRZIP" : "FolderClosed"); // TODO: ???
            child->setIconOpened(isSubVfs ? "IDP_DIRZIP" : "FolderOpened"); // TODO: ???
            dirTree.insertNodeInto(child, *parent, countAdded, true);
            countAdded += 1;

            if (fname.equalsString(nodeName, ignoreCase))
               nextChild = child;
         } 
         while (fs.findNext(hdir, fitem));
         if (hdir != 0)
            fs.findClose(hdir);
         if (nextChild == null)
            break; // Error: The current directory does no longer exist!
         parent = nextChild;
         path += nodeName;
         fs.slash(path);
      }
   }

   // Expand all nodes that forms the path to the current directory,
   // and make sure that node is scrolled into view.
   treeView.setSelectedNode(*parent);
}
