/* --------------------------------------------------------------------------
 *
 * Copyright (C) 2007 Leif Erik Larsen, Kjerringvik, Norway.
 *
 * This file is part of the Open Source Edition of Larsen Commander, as
 * available from http://home.online.no/~leifel/lcmd/.  This code is free 
 * software; you can redistribute it and/or modify it under the terms of 
 * the GNU General Public License version 3 only, as published by the 
 * Free Software Foundation.  
 *
 * This code is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 * version 3 at http://www.gnu.org/licenses/gpl-3.0.txt for more details 
 * (a copy is included in the LICENSE file that accompanied this code).
 *
 * ------------------------------------------------------------------------ */

#include "lcmd/LCmdVfsFilePreparator.h"
#include "lcmd/LCmdFilePanel.h"

#include "glib/gui/GProgressBar.h"
#include "glib/gui/GDialogPanel.h"

LCmdVfsFilePreparator::LCmdVfsFilePreparator ( LCmdFilePanel& fp )
                      :GWorkerThread("DlgVfsPrepareFile", 250),
                       physicalFileRep(null),
                       fileSize(0),
                       fp(fp),
                       vfs(fp.vfs.peek())
{
}

LCmdVfsFilePreparator::~LCmdVfsFilePreparator ()
{
   // Will be != null only if we still has the ownership. Normally we have not!
   delete physicalFileRep;
}

GVfs::File* LCmdVfsFilePreparator::prepareSelectedFile ( GWindow& parentWin )
{
   // Get the currently selected file within the file panel. 
   LCmdFileItem* fitem = fp.getCurItem();
   if (fitem == null) // If there is no current selected file.
      return null;
   if (fitem->isDirectory())
      return null;
   GVfs::File* f = fitem->getPhysicalFileRepresentation();
   if (f != null) // If the file item has already been "prepared".
      return f;

   // Prepare the file for physical action(s).
   fileSize = fitem->fileSize;
   originalFileName = fitem->getFullPath();
   f = prepareIt(parentWin);

   // Keep the prepared file in its associated file panel item,
   // for possible future re-use.
   fitem->setPhysicalFileRepresentation(f);
   return f; // Is null if user cancelled the operation.
}

GVfs::File* LCmdVfsFilePreparator::prepareSpecifiedFile ( GWindow& parentWin, const GString& path )
{
   GFileItem fitem(vfs, path);
   fileSize = fitem.fileSize;
   originalFileName = path;
   return prepareIt(parentWin);
}

GVfs::File* LCmdVfsFilePreparator::prepareIt ( GWindow& parentWin )
{
   if (vfs.isFilePreparationPossiblyLengthy())
   {
      workModal(parentWin);
      GVfs::File* f = physicalFileRep;
      physicalFileRep = null;
      return f; // Is null if user cancelled the operation.
   }
   else
   {
      return doIt();
   }
}

GVfs::File* LCmdVfsFilePreparator::doIt ()
{
   GString prefix = "lc-";
   return vfs.preparePhysicalFile(originalFileName, workStatus, prefix);
}

void LCmdVfsFilePreparator::runTheWorkerThread ( GWorkerThread& /*worker*/ )
{
   physicalFileRep = doIt();
}

void LCmdVfsFilePreparator::onWorkerThreadInitDialog ( GWorkerThread& worker, GDialogPanel& monitor )
{
   GProgressBar& p = dynamic_cast<GProgressBar&>(monitor.getComponentByID("ProgBar"));
   p.setMinValue(0);
   p.setMaxValue(1000); // Use "per thousand" instead of "per cent".
   p.setCurrentValue(0);
}

void LCmdVfsFilePreparator::onWorkerThreadCommand ( GWorkerThread& worker, GDialogPanel& monitor, const GString& cmd )
{
   if (cmd == "DLG_CANCEL")
   {
      workStatus.requestCancel();
      monitor.setComponentDisabled(cmd);
   }
}

void LCmdVfsFilePreparator::onWorkerThreadUpdateMonitor ( GWorkerThread& worker, GDialogPanel& monitor )
{
   longlong finishedCount = workStatus.getFinishedCount();
   int percentFinished = int((finishedCount * 1000) / fileSize);
   monitor.setComponentValue("ProgBar", percentFinished);
}
