UNIT tpz;
INTERFACE

  USES crt, dos, tpzasync, tpzvideo, tpzfiles, tpzunix, tpzcrc;

  FUNCTION zmodem_receive(path : string; comport : word; baudrate : longint):
        boolean;
  FUNCTION zmodem_send(pathname : string; lastfile : boolean; comport : word;
        baudrate : longint) : boolean;

IMPLEMENTATION

  CONST
    tpzver         = 'TPZ [Zmodem] 2.1';
    zbufsize       = 1024;
    zport          : word = 1;
    zbaud          : longint = 0;

  TYPE
    hdrtype        = ARRAY[0..3] OF byte;
    buftype        = ARRAY[0..1023] OF byte;

  CONST
    zpad           = 42;       { '*' }
    zdle           = 24;       { ^X  }
    zdlee          = 88;
    zbin           = 65;       { 'A' }
    zhex           = 66;       { 'B' }
    zbin32         = 67;       { 'C' }
    zrqinit        = 0;
    zrinit         = 1;
    zsinit         = 2;
    zack           = 3;
    zfile          = 4;
    zskip          = 5;
    znak           = 6;
    zabort         = 7;
    zfin           = 8;
    zrpos          = 9;
    zdata          = 10;
    zeof           = 11;
    zferr          = 12;
    zcrc           = 13;
    zchallenge     = 14;
    zcompl         = 15;
    zcan           = 16;
    zfreecnt       = 17;
    zcommand       = 18;
    zstderr        = 19;
    zcrce          = 104;      { 'h' }
    zcrcg          = 105;      { 'i' }
    zcrcq          = 106;      { 'j' }
    zcrcw          = 107;      { 'k' }
    zrub0          = 108;      { 'l' }
    zrub1          = 109;      { 'm' }
    zok            = 0;
    zerror         = -1;
    ztimeout       = -2;
    rcdo           = -3;
    fubar          = -4;
    gotor          = 256;
    gotcrce        = 360;      { 'h' OR 256 }
    gotcrcg        = 361;      { 'i' "   "  }
    gotcrcq        = 362;      { 'j' "   "  }
    gotcrcw        = 363;      { 'k' "   "  }
    gotcan         = 272;      { CAN OR  "  }

    { xmodem paramaters }
    enq            = 5;
    can            = 24;
    xoff           = 19;
    xon            = 17;
    soh            = 1;
    stx            = 2;
    eot            = 4;
    ack            = 6;
    nak            = 21;
    cpmeof         = 26;

    { byte positions }
    zf0            = 3;
    zf1            = 2;
    zf2            = 1;
    zf3            = 0;
    zp0            = 0;
    zp1            = 1;
    zp2            = 2;
    zp3            = 3;

    { bit masks for ZRINIT }
    canfdx         = 1;        { can handle full-duplex (yes for PC's)      }
    canovio        = 2;        { can overlay disk and serial I/O (ditto)    }
    canbrk         = 4;        { can send a break - True but superfluous    }
    cancry         = 8;        { can encrypt/decrypt - not defined yet      }
    canlzw         = 16;       { can LZ compress - not defined yet          }
    canfc32        = 32;       { can use 32 bit crc frame checks - true     }
    escall         = 64;       { escapes all control chars. NOT implemented }
    esc8           = 128;      { escapes the 8th bit. NOT implemented       }

    { bit masks for ZSINIT }
    tescctl        = 64;
    tesc8          = 128;

    { paramaters for ZFILE }

    { ZF0 }
    zcbin          = 1;
    zcnl           = 2;
    zcresum        = 3;

    { ZF1 }
    zmnew          = 1;    {I haven't implemented these as of yet - most are}
    zmcrc          = 2;    {superfluous on a BBS - Would be nice from a comm}
    zmapnd         = 3;    {programs' point of view however                 }
    zmclob         = 4;
    zmspars        = 5;
    zmdiff         = 6;
    zmprot         = 7;

    { ZF2 }
    ztlzw          = 1;    {encryption, compression and funny file handling }
    ztcrypt        = 2;    {flags - My docs (03/88) from OMEN say these have}
    ztrle          = 3;    {not been defined yet                            }

    { ZF3 }
    zcack1         = 1;    {God only knows...                               }

  VAR
    rxpos          : longint;    {file position received from Z_GetHeader}
    rxhdr          : hdrtype;    {receive header var}
    rxtimeout,
    rxtype,
    rxframeind     : integer;
    attn           : buftype;
    secbuf         : buftype;
    fname          : string;
    fmode          : integer;
    ftime, fsize   : longint;
    usecrc32       : boolean;
    zcps, zerrors  : word;
    txpos          : longint;
    txhdr          : hdrtype;
    ztime          : longint;

  CONST
    lastsent       : byte = 0;

  (* 1---------------1 *)

  FUNCTION z_settimer : longint;

    VAR
      l              : longint;
      h, m, s, x     : word;

    BEGIN (* z_settimer *)
    gettime(h, m, s, x); l := longint(h) * 3600;
    l := l + longint(m) * 60; l := l + longint(s);
    z_settimer := l
    END; (* z_settimer *)

  (* 1---------------1 *)

  FUNCTION z_filecrc32(VAR f : FILE) : longint;

    VAR
      fbuf           : buftype;
      crc            : longint;
      bread, n       : integer;

    BEGIN (* z_filecrc32 *)
{$I-}
    crc := $ffffffff; seek(f, 0);
    IF (ioresult <> 0) THEN {null};
    REPEAT
      blockread(f, fbuf, zbufsize, bread);
      FOR n := 0 TO (bread - 1) DO crc := updc32(fbuf[n], crc);
    UNTIL (bread < zbufsize) OR (ioresult <> 0);
    seek(f, 0);
{$I+}
    IF (ioresult <> 0) THEN {null};
    z_filecrc32 := crc;
    END; (* z_filecrc32 *)

  (* 1---------------1 *)

  FUNCTION z_getbyte(tenths : integer) : integer;
  (* Reads a byte from the modem - Returns RCDO if *)
  (* no carrier, or ZTIMEOUT if nothing received   *)
  (* within 'tenths' of a second.                  *)

    VAR
      n              : integer;

    BEGIN (* z_getbyte *)
    REPEAT
      IF (NOT z_carrier) THEN BEGIN
        z_getbyte := rcdo;       { nobody to talk to }
        exit; END;                                        (* <<-- EXIT *)
      IF (z_charavail) THEN BEGIN
        z_getbyte := z_receivebyte; { got character }
        exit; END;                                        (* <<-- EXIT *)
      dec(tenths);                { dec. the count    }
      delay(100);                 { pause 1/10th sec. }
    UNTIL (tenths <= 0);
    z_getbyte := ztimeout;        { timed out }
    END; (* z_getbyte *)

  (* 1---------------1 *)

  FUNCTION z_qk_read : integer;
  (* Just like Z_GetByte, but timeout value is in *)
  (* global var rxtimeout.                        *)

    BEGIN (* z_qk_read *)
    z_qk_read := z_getbyte(rxtimeout);
    END; (* z_qk_read *)

  (* 1---------------1 *)

  FUNCTION z_timedread : integer;
  (* A Z_qk_read, that strips parity and *)
  (* ignores XON/XOFF characters.        *)

    VAR
      done           : boolean;
      c              : integer;

    BEGIN (* z_timedread *)
    done := false;
    REPEAT
      c := z_qk_read AND $ff7f    { strip parity }
    UNTIL (c < 0) OR
          (NOT (lo(c) IN [17, 19])); { wait for other than XON/XOFF }
    z_timedread := c;
    END; (* z_timedread *)

  (* 1---------------1 *)

  PROCEDURE z_sendcan;
  (* Send a zmodem CANcel sequence to the other guy *)
  (* 8 CANs and 8 backspaces                        *)

    VAR
      n              : byte;

    BEGIN (* z_sendcan *)
    z_clearoutbound;             { spare them the junk }
    FOR n := 1 TO 8 DO BEGIN
      { the pause seems to make reception of the sequence more reliable }
      z_sendbyte(can); delay(100); END;
    FOR n := 1 TO 10 DO z_sendbyte(8);
    END; (* z_sendcan *)

  (* 1---------------1 *)

  PROCEDURE z_putstring(VAR p : buftype);
  (* Outputs an ASCII-Z type string (null terminated) *)
  (* Processes meta characters 221 (send break) and   *)
  (* 222 (2 second delay).                            *)

    VAR
      n              : integer;

    BEGIN (* z_putstring *)
    n := 0;
    WHILE (n < zbufsize) AND (p[n] <> 0) DO BEGIN
      CASE p[n] OF
  221 : z_sendbreak;
  222 : delay(2000);
      ELSE z_sendbyte(p[n]);
        END; (* case *)
      inc(n); END;
    END; (* z_putstring *)

  (* 1---------------1 *)

  PROCEDURE z_puthex(b : byte);
  (* Output a byte as two hex digits (in ASCII) *)
  (* Uses lower case to avoid confusion with    *)
  (* escaped control characters.                *)

    CONST
      hex            : ARRAY[0..15] OF char = '0123456789abcdef';

    BEGIN (* z_puthex *)
    z_sendbyte(ord(hex[b shr 4]));    { high nybble }
    z_sendbyte(ord(hex[b AND $0f]));  { low nybble  }
    END; (* z_puthex *)

  (* 1---------------1 *)

  PROCEDURE z_sendhexheader(htype : byte; VAR hdr : hdrtype);
  (* Sends a zmodem hex type header *)

    VAR
      crc            : word;
      n, i           : integer;

    BEGIN (* z_sendhexheader *)
    z_sendbyte(zpad);            { '*' }
    z_sendbyte(zpad);            { '*' }
    z_sendbyte(zdle);            { 24  }
    z_sendbyte(zhex);            { 'B' }
    z_puthex(htype); crc := updcrc(htype, 0);
    FOR n := 0 TO 3 DO BEGIN
      z_puthex(hdr[n]); crc := updcrc(hdr[n], crc); END;
    crc := updcrc(0, crc); crc := updcrc(0, crc);
    z_puthex(lo(crc shr 8)); z_puthex(lo(crc));
    z_sendbyte(13);              { make it readable to the other end }
    z_sendbyte(10);              { just in case                      }
    IF (htype <> zfin) AND (htype <> zack) THEN
      z_sendbyte(17);            { Prophylactic XON to assure flow   }
    IF (NOT z_carrier) THEN z_clearoutbound;
    END; (* z_sendhexheader *)

  (* 1---------------1 *)

  FUNCTION z_pulllongfromheader(VAR hdr : hdrtype) : longint;
  (* Stuffs a longint into a header variable - N.B. - bytes are REVERSED! *)

    VAR
      l              : longint;

    BEGIN (* z_pulllongfromheader *)
    l := hdr[zp3];               { hard coded for efficiency }
    l := (l shl 8) OR hdr[zp2];
    l := (l shl 8) OR hdr[zp1];
    l := (l shl 8) OR hdr[zp0];
    z_pulllongfromheader := l;
    END; (* z_pulllongfromheader *)

  (* 1---------------1 *)

  PROCEDURE z_putlongintoheader(l : longint);
  (* Reverse of above *)

    BEGIN (* z_putlongintoheader *)
    txhdr[zp0] := byte(l); txhdr[zp1] := byte(l shr 8);
    txhdr[zp2] := byte(l shr 16);
    txhdr[zp3] := byte(l shr 24);
    END; (* z_putlongintoheader *)

  (* 1---------------1 *)

  FUNCTION z_getzdl : integer;
  (* Gets a byte and processes for ZMODEM escaping or CANcel sequence *)

    VAR
      c, d           : integer;

    BEGIN (* z_getzdl *)
    IF (NOT z_carrier) THEN BEGIN
      z_getzdl := rcdo; exit; END;                        (* <<-- EXIT *)
    c := z_qk_read;
    if (C <> ZDLE) then begin
      z_getzdl := c; exit; END;     {got ZDLE or 1st CAN} (* <<--EXIT *)
    c := z_qk_read;
    IF (c = can) THEN BEGIN         {got 2nd CAN}
      c := z_qk_read;
      IF (c = can) THEN BEGIN       {got 3rd CAN}
        c := z_qk_read;
        IF (c = can) THEN           {got 4th CAN}
          c := z_qk_read; END;
      END;

    { Flags set in high byte }
    CASE c OF
can : z_getzdl := gotcan;              {got 5th CAN}
zcrce,                                 {got a frame end marker}
zcrcg,
zcrcq,
zcrcw:
      z_getzdl := (c OR gotor);
zrub0:
      z_getzdl := $007f;                {got an ASCII DELete}
zrub1:
      z_getzdl := $00ff;                {any parity         }
    ELSE BEGIN
      IF (c < 0) THEN z_getzdl := c
      ELSE IF ((c AND $60) = $40) THEN  {make sure it was a valid escape}
        z_getzdl := c xor $40
      ELSE z_getzdl := zerror; END;
      END; (* case *)
    END; (* get_zdl *)

  (* 1---------------1 *)

  FUNCTION z_gethex : integer;
  (* Get a byte that has been received as two ASCII hex digits *)

    VAR
      c, n           : integer;

    BEGIN (* z_gethex *)
    n := z_timedread;
    IF (n < 0) THEN BEGIN
      z_gethex := n; exit; END;                           (* <<-- EXIT *)
    n := n - $30;                {build the high nybble}
    IF (n > 9) THEN n := n - 39;
    IF (n AND $fff0 <> 0) THEN BEGIN
      z_gethex := zerror; exit; END;                      (* <<-- EXIT *)
    c := z_timedread;
    IF (c < 0) THEN BEGIN
      z_gethex := c; exit; END;                           (* <<-- EXIT *)
    c := c - $30;                {now the low nybble}
    IF (c > 9) THEN c := c - 39;
    IF (c AND $fff0 <> 0) THEN BEGIN
      z_gethex := zerror; exit; END;                      (* <<-- EXIT *)
    z_gethex := (n shl 4) OR c;   {Insert tab 'A' in slot 'B'...}
    END; (* z_gethex *)

  (* 1---------------1 *)

  FUNCTION z_gethexheader(VAR hdr : hdrtype) : integer;
  (* Receives a zmodem hex type header *)

    VAR
      crc            : word;
      c, n           : integer;

    BEGIN (* z_gethexheader *)
    c := z_gethex;
    IF (c < 0) THEN BEGIN
      z_gethexheader := c; exit; END;                     (* <<-- EXIT *)
    rxtype := c;                 {get the type of header}
    crc := updcrc(rxtype, 0);
    FOR n := 0 TO 3 DO BEGIN           {get the 4 bytes}
      c := z_gethex;
      IF (c < 0) THEN BEGIN
        z_gethexheader := c; exit; END;                   (* <<-- EXIT *)
      hdr[n] := lo(c); crc := updcrc(lo(c), crc); END;
    c := z_gethex;
    IF (c < 0) THEN BEGIN
      z_gethexheader := c; exit; END;                     (* <<-- EXIT *)
    crc := updcrc(lo(c), crc); c := z_gethex;
    IF (c < 0) THEN BEGIN
      z_gethexheader := c; exit; END;                     (* <<-- EXIT *)
    crc := updcrc(lo(c), crc);   {check the CRC}
    IF (crc <> 0) THEN BEGIN
      inc(zerrors); z_errors(zerrors);
      z_gethexheader := zerror; exit; END;                (* <<-- EXIT *)
    IF (z_getbyte(1) = 13) THEN  {throw away CR/LF}
      c := z_getbyte(1);
    z_gethexheader := rxtype;
    END; (* z_gethexheader *)

  (* 1---------------1 *)

  FUNCTION z_getbinaryheader(VAR hdr : hdrtype) : integer;
  (* Same as above, but binary with 16 bit CRC *)

    VAR
      crc            : word;
      c, n           : integer;

    BEGIN (* z_getbinaryheader *)
    c := z_getzdl;
    IF (c < 0) THEN BEGIN
      z_getbinaryheader := c; exit; END;                  (* <<-- EXIT *)
    rxtype := c; crc := updcrc(rxtype, 0);
    FOR n := 0 TO 3 DO BEGIN
      c := z_getzdl;
      IF (hi(c) <> 0) THEN BEGIN
        z_getbinaryheader := c; exit; END;                (* <<-- EXIT *)
      hdr[n] := lo(c); crc := updcrc(lo(c), crc); END;
    c := z_getzdl;
    IF (hi(c) <> 0) THEN BEGIN
      z_getbinaryheader := c; exit; END;                  (* <<-- EXIT *)
    crc := updcrc(lo(c), crc); c := z_getzdl;
    IF (hi(c) <> 0) THEN BEGIN
      z_getbinaryheader := c; exit; END;                  (* <<-- EXIT *)
    crc := updcrc(lo(c), crc);
    IF (crc <> 0) THEN BEGIN
      inc(zerrors); z_errors(zerrors); exit; END;         (* <<-- EXIT *)
    z_getbinaryheader := rxtype;
    END; (* z_getbinaryheader *)

  (* 1---------------1 *)

  FUNCTION z_getbinaryhead32(VAR hdr : hdrtype) : integer;
  (* Same as above but with 32 bit CRC *)

    VAR
      crc            : longint;
      c, n           : integer;

    BEGIN (* z_getbinaryhead32 *)
    c := z_getzdl;
    IF (c < 0) THEN BEGIN
      z_getbinaryhead32 := c; exit; END;                  (* <<-- EXIT *)
    rxtype := c; crc := updc32(rxtype, $ffffffff);
    FOR n := 0 TO 3 DO BEGIN
      c := z_getzdl;
      IF (hi(c) <> 0) THEN BEGIN
        z_getbinaryhead32 := c; exit; END;                (* <<-- EXIT *)
      hdr[n] := lo(c); crc := updc32(lo(c), crc); END;
    FOR n := 0 TO 3 DO BEGIN
      c := z_getzdl;
      IF (hi(c) <> 0) THEN BEGIN
        z_getbinaryhead32 := c; exit; END;                (* <<-- EXIT *)
      crc := updc32(lo(c), crc); END;
    IF (crc <> $debb20e3) THEN BEGIN  {this is the polynomial value}
      inc(zerrors); z_errors(zerrors);
      z_getbinaryhead32 := zerror; exit; END;             (* <<-- EXIT *)
    z_getbinaryhead32 := rxtype;
    END; (* z_getbinaryhead32 *)

  (* 1---------------1 *)

  FUNCTION z_getheader(VAR hdr : hdrtype) : integer;
  (* Use this routine to get a header - it will figure out  *)
  (* what type it is getting (hex, bin16 or bin32) and call *)
  (* the appropriate routine.                               *)

    LABEL 10 {gotcan},         {sorry, but it's actually easier to }
          11 {again},          {follow, and lots more efficient    }
          12 {agn2},           {this way...                        }
          13 {splat},
          14 {done};

    VAR
      c, n, cancount : integer;

    BEGIN (* z_getheader *)
    n := zbaud * 2;              {A guess at the # of garbage characters}
    cancount := 5;               {to expect.                            }
    usecrc32 := false;           {assume 16 bit until proven otherwise  }

11:    {again}
    IF (keypressed) THEN              {check for operator panic}
      IF (readkey = #27) THEN BEGIN   {in the form of ESCape   }
        z_sendcan;                    {tell the other end, the operator,}
        z_message('Cancelled from keyboard');    {and the rest of the   }
        z_getheader := zcan;                     {routines to forget it.}
        exit; END;                                        (* <<-- EXIT *)
    rxframeind := 0; rxtype := 0; c := z_timedread;
    CASE c OF
zpad: (* nothing *);        {we want this! - all headers begin with '*'.}
rcdo,
ztimeout:
      GOTO 14 {done};
can:  BEGIN
10:   {gotcan}
      dec(cancount);
      IF (cancount < 0) THEN BEGIN
        c := zcan; GOTO 14 {done}; END;                   (* <<-- DONE *)
      c := z_getbyte(1);
      CASE c OF
ztimeout:
        GOTO 11 {again};                                  (* <<-- AGAIN *)
zcrcw:  BEGIN
        c := zerror; GOTO 14 {done};                      (* <<-- DONE *)
        END;
rcdo:   GOTO 14 {done};                                   (* <<-- DONE *)
can:    BEGIN
        dec(cancount);
        IF (cancount < 0) THEN BEGIN
          c := zcan; GOTO 14 {done}; END;                 (* <<-- DONE *)
        GOTO 11;                                          (* <<-- AGAIN *)
        END;
      ELSE {fallthru};
        END; (* case *)
      END; {outer case can:}
    ELSE
12: {agn2}
      BEGIN
      dec(n);
      IF (n < 0) THEN BEGIN
        inc(zerrors); z_errors(zerrors);
        z_message('Header is FUBAR');
        z_getheader := zerror; exit; END;                 (* <<-- EXIT *)
      IF (c <> can) THEN cancount := 5;
      GOTO 11; END;                                       (* <<-- AGAIN *)
      END; {only falls thru if ZPAD - anything else is trash}
    cancount := 5;

13: {splat}
    c := z_timedread;
    CASE c OF
zdle: (* nothing *);        {this is what we want!}
zpad: GOTO 13; {junk or second '*' of a hex header}       (* <<-- SPLAT *)
rcdo,
ztimeout:
      GOTO 14;                                            (* <<-- DONE *)
    ELSE GOTO 12;                                         (* <<-- AGN2 *)
      END; (* case *) {only falls thru if ZDLE}
    c := z_timedread;
    CASE c OF
zbin32:
      BEGIN
      rxframeind := zbin32;      {using 32 bit CRC}
      c := z_getbinaryhead32(hdr)
      END;
zbin: BEGIN
      rxframeind := zbin;        {bin with 16 bit CRC}
      c := z_getbinaryheader(hdr)
      END;
zhex: BEGIN
      rxframeind := zhex;        {hex}
      c := z_gethexheader(hdr)
      END;
can:  GOTO 10;                                            (* <<-- GOTCAN *)
rcdo,
ztimeout:
      GOTO 14;                                            (* <<-- DONE *)
    ELSE GOTO 12;                                         (* <<-- AGN2 *)
      END; (* case *)  {only falls thru if we got ZBIN, ZBIN32 or ZHEX}
    rxpos := z_pulllongfromheader(hdr); {set rxpos just in case this}

14: {done}                        {header has file position  }
    z_getheader := c;             {info (i.e. : ZRPOS, etc.) }
    END; (* z_getheader *)

  (* 1---------------1 *)

  (***************************************************)
  (* RECEIVE FILE ROUTINES                           *)
  (***************************************************)

  CONST
    zattnlen       = 32;         {max length of attention string}
    lastwritten    : byte = 0;

  VAR
    t              : longint;
    rzbatch        : boolean;
    outfile        : FILE;       {this is the file}

    tryzhdrtype    : byte;
    rxcount        : integer;
    filestart      : longint;
    isbinary,
    eofseen        : boolean;
    zconv          : byte;
    zrxpath        : string;

  (* 1---------------1 *)

  FUNCTION rz_receiveda32(VAR buf : buftype; blength : integer) : integer;
  (* Get a 32 bit CRC data block *)

    LABEL 20 {crcfoo};

    VAR
      c, d, n        : integer;
      crc            : longint;
      done           : boolean;

    BEGIN (* rz_receiveda32 *)
    usecrc32 := true; crc := $ffffffff; rxcount := 0; done := false;
    REPEAT
      c := z_getzdl;
      IF (hi(c) <> 0) THEN BEGIN
20:     {crcfoo}
        CASE c OF
gotcrce,
gotcrcg,
gotcrcq,
gotcrcw:  BEGIN
          d := c; crc := updc32(lo(c), crc);
          FOR n := 0 TO 3 DO BEGIN
            c := z_getzdl;
            IF (hi(c) <> 0) THEN GOTO 20;                 (* <<-- CRCFOO *)
            crc := updc32(lo(c), crc); END;
          IF (crc <> $debb20e3) THEN BEGIN
            inc(zerrors); z_errors(zerrors);
            rz_receiveda32 := zerror; END
          ELSE rz_receiveda32 := d;
          done := true;
          END;
gotcan:   BEGIN
          rz_receiveda32 := zcan; done := true;
          END;
ztimeout: BEGIN
          rz_receiveda32 := c; done := true;
          END;
rcdo:     BEGIN
          rz_receiveda32 := c; done := true;
          END;
        ELSE BEGIN
          z_message('Debris'); z_clearinbound;
          rz_receiveda32 := c; done := true; END;
          END; (* case *)
        END;
      IF (NOT done) THEN BEGIN
        dec(blength);
        IF (blength < 0) THEN BEGIN
          z_message('Long packet');
          rz_receiveda32 := zerror; done := true; END;
        buf[integer(rxcount)] := lo(c); inc(rxcount);
        crc := updc32(lo(c), crc); END;
    UNTIL done;
    END; (* rz_receiveda32 *)

  (* 1---------------1 *)

  FUNCTION rz_receivedata(VAR buf : buftype; blength : integer) : integer;
  (* get a 16 bit CRC data block *)

    LABEL 30 {crcfoo};

    VAR
      c, d           : integer;
      crc            : word;
      done           : boolean;

    BEGIN (* rz_receiveddata *)
    IF (rxframeind = zbin32) THEN BEGIN
      z_showcheck(true);
      rz_receivedata := rz_receiveda32(buf, blength);
      exit; END;                                          (* <<-- EXIT *)
    z_showcheck(false); crc := 0; rxcount := 0; done := false;
    REPEAT
      c := z_getzdl;
      IF (hi(c) <> 0) THEN BEGIN
30: {crcfoo}
        CASE c OF
gotcrce,
gotcrcg,
gotcrcq,
gotcrcw:  BEGIN
          d := c; crc := updcrc(lo(c), crc);
          c := z_getzdl;
          IF (hi(c) <> 0) THEN GOTO 30;                   (* <<-- CRCFOO *)
          crc := updcrc(lo(c), crc); c := z_getzdl;
          IF (hi(c) <> 0) THEN GOTO 30;                   (* <<-- CRCFOO *)
          crc := updcrc(lo(c), crc);
          IF (crc <> 0) THEN BEGIN
            inc(zerrors); z_errors(zerrors);
            rz_receivedata := zerror; done := true; END;
          rz_receivedata := d; done := true;
          END;
gotcan:   BEGIN
          z_message('Got CANned');
          rz_receivedata := zcan; done := true;
          END;
ztimeout: BEGIN
          rz_receivedata := c; done := true;
          END;
rcdo:     BEGIN
          z_message('Lost carrier'); rz_receivedata := c; done := true;
          END;
        ELSE BEGIN
          z_message('Debris'); z_clearinbound;
          rz_receivedata := c; done := true; END;
          END; (* case *)
        END;
      IF (NOT done) THEN BEGIN
        dec(blength);
        IF (blength < 0) THEN BEGIN
          z_message('Long packet');
          rz_receivedata := zerror; done := true; END;
        buf[integer(rxcount)] := lo(c); inc(rxcount);
        crc := updcrc(lo(c), crc); END;
    UNTIL done;
    END; (* rz_receivedata *)

  (* 1---------------1 *)

  PROCEDURE rz_ackbibi;
  (* ACKnowledge the other ends request to terminate cleanly *)

    VAR
      n              : integer;

    BEGIN (* rz_ackbibi *)
    z_putlongintoheader(rxpos); n := 4; z_clearinbound;
    REPEAT
      z_sendhexheader(zfin, txhdr);
      CASE z_getbyte(20) OF
ztimeout,
rcdo:   exit;                                             (* <<-- EXIT *)
79:     BEGIN
        IF (z_getbyte(10) = 79) THEN {null};
        z_clearinbound; exit;                             (* <<-- EXIT *)
        END;
      ELSE z_clearinbound; dec(n);
        END (* case *)
    UNTIL (n <= 0);
    END; (* rz_ackbibi *)

  (* 1---------------1 *)

  FUNCTION rz_initreceiver : integer;

    LABEL 40 {again};

    VAR
      c, n, errors   : integer;

    BEGIN (* rz_initreceiver *)
    fillchar(attn, sizeof(attn), 0); zerrors := 0;
    FOR n := 10 DOWNTO 0 DO BEGIN
      IF (NOT z_carrier) THEN BEGIN
        z_message('Lost carrier');
        rz_initreceiver := zerror; exit; END;             (* <<-- EXIT *)
      z_putlongintoheader(longint(0));
      txhdr[zf0] := canfdx OR canovio OR canfc32 OR canbrk;
                    {Full dplx, overlay I/O and CRC32}
      z_sendhexheader(tryzhdrtype, txhdr);
      IF (tryzhdrtype = zskip) THEN tryzhdrtype := zrinit;

40: {again}
      c := z_getheader(rxhdr); z_frame(c);
      CASE c OF
zfile:  BEGIN
        zconv := rxhdr[zf0]; tryzhdrtype := zrinit;
        c := rz_receivedata(secbuf, zbufsize);
        z_frame(c);
        IF (c = gotcrcw) THEN BEGIN
          rz_initreceiver := zfile; exit; END;            (* <<-- EXIT *)
        z_sendhexheader(znak, txhdr); GOTO 40;            (* <<-- AGAIN *)
        END;
zsinit: BEGIN
        c := rz_receivedata(attn, zbufsize); z_frame(c);
        IF (c = gotcrcw) THEN z_sendhexheader(zack, txhdr)
        ELSE z_sendhexheader(znak, txhdr);
        GOTO 40;                                          (* <<-- AGAIN *)
        END;
zfreecnt:
        BEGIN
        z_putlongintoheader(diskfree(0));
        z_sendhexheader(zack, txhdr); GOTO 40;            (* <<-- AGAIN *)
        END;
zcommand:
        BEGIN
        c := rz_receivedata(secbuf, zbufsize);
        z_frame(c);
        IF (c = gotcrcw) THEN BEGIN
          z_putlongintoheader(longint(0));
          REPEAT
            z_sendhexheader(zcompl, txhdr); inc(errors)
          UNTIL (errors > 10) OR (z_getheader(rxhdr) = zfin);
          rz_ackbibi; rz_initreceiver := zcompl;
          exit; END;                                      (* <<-- EXIT *)
        z_sendhexheader(znak, txhdr); GOTO 40;            (* <<-- AGAIN *)
        END;
zcompl,
zfin:   BEGIN
        rz_initreceiver := zcompl; exit;                  (* <<-- EXIT *)
        END;
zcan,
rcdo:   BEGIN
        rz_initreceiver := c; exit;                       (* <<-- EXIT *)
        END;
        END; (* case *)
      END; (* for *)
    z_message('Timeout'); rz_initreceiver := zerror;
    END; (* rz_initreceiver *)

  (* 1---------------1 *)

  FUNCTION rz_getheader : integer;

    VAR
      e, p, n, i     : integer;
      multiplier     : longint;
      s              : string;
      ttime, tsize   : longint;
      tname          : string;

    BEGIN (* rz_getheader *)
    isbinary := true;            {Force the issue!}
    fsize := longint(0); p := 0; s := '';
    WHILE (p < 255) AND (secbuf[p] <> 0) DO BEGIN
      s := s + upcase(chr(secbuf[p])); inc(p); END;
    inc(p);
    (* get rid of drive & path specifiers *)
    WHILE (pos(':', s) > 0) DO delete(s, 1, pos(':', s));
    WHILE (pos('\', s) > 0) DO delete(s, 1, pos('\', s));
    fname := s;

    (**** done with name ****)

    fsize := longint(0);
    WHILE (p < zbufsize) AND
          (secbuf[p] <> $20) AND (secbuf[p] <> 0) DO BEGIN
      fsize := (fsize *10) + ord(secbuf[p]) - $30;
      inc(p); END;
    inc(p);

    (**** done with size ****)

    s := '';
    WHILE (p < zbufsize) AND (secbuf[p] IN [$30..$37]) DO BEGIN
      s := s + chr(secbuf[p]); inc(p); END;
    inc(p); ftime := z_fromunixdate(s);

    (**** done with time ****)

    IF (z_findfile(zrxpath+fname, tname, tsize, ttime)) THEN BEGIN
      IF (zconv = zcresum) AND (fsize > tsize) THEN BEGIN
        filestart := tsize;
        IF (NOT z_openfile(outfile, zrxpath + fname)) THEN BEGIN
          z_message('Error opening '+fname);
          rz_getheader := zerror; exit; END;              (* <<-- EXIT *)
        IF (NOT z_seekfile(outfile, tsize)) THEN BEGIN
          z_message('Error positioning file');
          rz_getheader := zerror; exit; END;              (* <<-- EXIT *)
        z_message('Recovering'); END
      ELSE BEGIN
        z_showname(fname);
        z_message('File is already complete');
        rz_getheader := zskip; exit; END;                 (* <<-- EXIT *)
      END
    ELSE BEGIN
      filestart := 0;
      IF (NOT z_makefile(outfile, zrxpath + fname)) THEN BEGIN
        z_message('Unable to create '+fname);
        rz_getheader := zerror; exit; END;                (* <<-- EXIT *)
      END;
    z_showname(fname); z_showsize(fsize);
    z_showtransfertime(fsize, zbaud);
    rz_getheader := zok;
    END; (* rz_getheader *)

  (* 1---------------1 *)

  FUNCTION rz_savetodisk(VAR rxbytes : longint) : integer;

    BEGIN (* rz_savetodisk *)
    IF (keypressed) THEN
      IF (readkey = #27) THEN BEGIN
        z_message('Aborted from keyboard'); z_sendcan;
        rz_savetodisk := zerror; exit; END;               (* <<-- EXIT *)
    IF (NOT z_writefile(outfile, secbuf, rxcount)) THEN BEGIN
      z_message('Disk write error');
      rz_savetodisk := zerror; END
    ELSE rz_savetodisk := zok;
    rxbytes := rxbytes + rxcount;
    END; (* rz_savetodisk *)

  (* 1---------------1 *)

  FUNCTION rz_receivefile : integer;

    LABEL 59 {err}, 50 {nxthdr}, 51 {moredata};

    VAR
      c, n           : integer;
      rxbytes        : longint;
      sptr           : string;
      done           : boolean;

    BEGIN (* rz_receivefile *)
    zerrors := 0; done := false; eofseen := false;
    c := rz_getheader;
    IF (c <> zok) THEN BEGIN
      IF (c = zskip) THEN tryzhdrtype := zskip;
      rz_receivefile := c; exit; END;                     (* <<-- EXIT *)
    c := zok; n := 10; rxbytes := filestart;
    rxpos := filestart; ztime := z_settimer; zcps := 0;
    REPEAT
      z_putlongintoheader(rxbytes);
      z_sendhexheader(zrpos, txhdr);
50:   (* NXTHDR *)
      c := z_getheader(rxhdr); z_frame(c);
      CASE c OF
zdata:  BEGIN
        IF (rxpos <> rxbytes) THEN BEGIN
          dec(n); inc(zerrors); z_errors(zerrors);
          IF (n < 0) THEN GOTO 59;                        (* <<-- ERR *)
          z_message('Bad position'); z_putstring(attn); END
        ELSE BEGIN
51:       (* MOREDATA *)
          c := rz_receivedata(secbuf, zbufsize);
          z_frame(c);
          CASE c OF
      zcan,
      rcdo: GOTO 59;                                      (* <<-- ERR *)
    zerror: BEGIN
            dec(n); inc(zerrors); z_errors(zerrors);
            IF (n < 0) THEN GOTO 59;                      (* <<-- ERR *)
            z_putstring(attn);
            END;
  ztimeout: BEGIN
            dec(n);
            IF (n < 0) THEN GOTO 59;                      (* <<-- ERR *)
            END;
   gotcrcw: BEGIN
            n := 10; c := rz_savetodisk(rxbytes);
            IF (c <> zok) THEN BEGIN
              rz_receivefile := c; exit; END;             (* <<-- EXIT *)
            z_showloc(rxbytes);
            z_putlongintoheader(rxbytes);
            z_sendhexheader(zack, txhdr); GOTO 50;        (* <<-- NXTHDR *)
            END;
   gotcrcq: BEGIN
            n := 10; c := rz_savetodisk(rxbytes);
            IF (c <> zok) THEN BEGIN
              rz_receivefile := c; exit; END;             (* <<-- EXIT *)
            z_showloc(rxbytes);
            z_putlongintoheader(rxbytes);
            z_sendhexheader(zack, txhdr); GOTO 51;        (* <<-- MOREDATA *)
            END;
   gotcrcg: BEGIN
            n := 10; c := rz_savetodisk(rxbytes);
            IF (c <> zok) THEN BEGIN
              rz_receivefile := c; exit; END;             (* <<-- EXIT *)
            z_showloc(rxbytes); GOTO 51;                  (* <<-- MOREDATA *)
            END;
   gotcrce: BEGIN
            n := 10; c := rz_savetodisk(rxbytes);
            IF (c <> zok) THEN BEGIN
              rz_receivefile := c; exit; END;             (* <<-- EXIT *)
            z_showloc(rxbytes); GOTO 50;                  (* <<-- NXTHDR *)
            END;
            END  {case}
          END;
        END; {ZDATA case}
znak,
ztimeout:
        BEGIN
        dec(n);
        IF (n < 0) THEN GOTO 59;                          (* <<-- ERR *)
        z_showloc(rxbytes);
        END;
zfile:  BEGIN
        c := rz_receivedata(secbuf, zbufsize);
        z_frame(c);
        END;
zeof:   IF (rxpos = rxbytes) THEN BEGIN
          rz_receivefile := c; exit; END                  (* <<-- EXIT *)
        ELSE GOTO 50;                                     (* <<-- NXTHDR *)
zerror: BEGIN
        dec(n);
        IF (n < 0) THEN GOTO 59;                          (* <<-- ERR *)
        z_showloc(rxbytes); z_putstring(attn);
        END;
      ELSE BEGIN
        c := zerror; GOTO 59; END;                        (* <<-- ERR *)
        END  {case}
    UNTIL (NOT done);
59: rz_receivefile := zerror;
    END; (* rz_receivefile *)

  (* 1---------------1 *)

  FUNCTION rz_receivebatch : integer;

    VAR
      s              : string;
      c              : integer;
      done           : boolean;

    BEGIN (* rz_receivebatch *)
    z_message('Receiving...'); done := false;
    WHILE (NOT done) DO BEGIN
      IF NOT (z_carrier) THEN BEGIN
        rz_receivebatch := zerror; exit; END;             (* <<-- EXIT *)
      c := rz_receivefile;
      zcps := fsize DIV (z_settimer - ztime);
      z_frame(c); z_setftime(outfile, ftime);
      z_closefile(outfile); str(zcps : 4, s);
      z_message(s+' cps');
      CASE c OF
zeof,
zskip:  BEGIN
        c := rz_initreceiver; z_frame(c);
        CASE c OF
   zfile: {null};
  zcompl: BEGIN
          rz_ackbibi; rz_receivebatch := zok; exit
          END;
        ELSE BEGIN
          rz_receivebatch := zerror; exit; END;           (* <<-- EXIT *)
          END; (* inner case *)
        END
      ELSE BEGIN
        rz_receivebatch := c; exit; END;
        END; {case}
      END; {while}
    END; (* rz_receivebatch *)

  (* 1---------------1 *)

  FUNCTION zmodem_receive(path : string; comport : word; baudrate : longint)
                          : boolean;
    VAR
      i              : integer;

    BEGIN (* zmodem_receive *)
    zbaud := baudrate; zport := comport;
    z_openwindow(tpzver); z_message('Initializing...');
    IF (NOT z_asyncon(comport, baudrate)) THEN BEGIN
      clrscr; writeln('Unable to open:');
      writeln('Port: ', comport);
      writeln('Baud: ', baudrate); delay(2000);
      z_closewindow; zmodem_receive := false; exit; END;  (* <<-- EXIT *)
    zrxpath := path;
    IF (zrxpath[length(zrxpath)] <> '\') AND (zrxpath <> '') THEN
      zrxpath := zrxpath + '\';
    rxtimeout := 100; tryzhdrtype := zrinit;
    i := rz_initreceiver;
    IF (i = zcompl) OR ((i = zfile) AND ((rz_receivebatch) = zok)) THEN BEGIN
      z_message('Restoring async params'); z_asyncoff;
      z_closewindow; zmodem_receive := true; END
    ELSE BEGIN
      z_clearoutbound; z_message('Sending CAN');
      z_sendcan; z_message('Restoring async params');
      z_asyncoff; z_closewindow;
      zmodem_receive := false; END;
    END; (* zmodem_receive *)

  (* 1---------------1 *)

  (*######### SEND ROUTINES #####################################*)

  VAR
    infile         : FILE;
    strtpos        : longint;
    rxbuflen       : integer;
    txbuf          : buftype;
    blkred         : integer;

  (* 1---------------1 *)

  PROCEDURE sz_z_sendbyte(b : byte);

    BEGIN (* sz_z_sendbyte *)
    IF ((b AND $7f) IN [16, 17, 19, 24]) OR
       (((b AND $7f) = 13) AND ((lastsent AND $7f) = 64)) THEN BEGIN
      z_sendbyte(zdle); lastsent := (b xor 64); END
    ELSE lastsent := b;
    z_sendbyte(lastsent);
    END; (* sz_z_sendbyte *)

  (* 1---------------1 *)

  PROCEDURE sz_sendbinaryhead32(htype : byte; VAR hdr : hdrtype);

    VAR
      crc            : longint;
      n              : integer;

    BEGIN (* sz_sendbinaryhead32 *)
    z_sendbyte(zpad); z_sendbyte(zdle);
    z_sendbyte(zbin32); sz_z_sendbyte(htype);
    crc := updc32(htype, $ffffffff);
    FOR n := 0 TO 3 DO BEGIN
      sz_z_sendbyte(hdr[n]); crc := updc32(hdr[n], crc); END;
    crc := (NOT crc);
    FOR n := 0 TO 3 DO BEGIN
      sz_z_sendbyte(byte(crc)); crc := (crc shr 8); END;
    IF (htype <> zdata) THEN delay(500);
    END; (* sz_sendbinaryhead32 *)

  (* 1---------------1 *)

  PROCEDURE sz_sendbinaryheader(htype : byte; VAR hdr : hdrtype);

    VAR
      crc            : word;
      n              : integer;

    BEGIN (* sz_sendbinaryheader *)
    IF (usecrc32) THEN BEGIN
      sz_sendbinaryhead32(htype, hdr); exit; END;         (* <<-- EXIT *)
    z_sendbyte(zpad); z_sendbyte(zdle);
    z_sendbyte(zbin); sz_z_sendbyte(htype);
    crc := updcrc(htype, 0);
    FOR n := 0 TO 3 DO BEGIN
      sz_z_sendbyte(hdr[n]); crc := updcrc(hdr[n], crc); END;
    crc := updcrc(0, crc); crc := updcrc(0, crc);
    sz_z_sendbyte(lo(crc shr 8)); sz_z_sendbyte(lo(crc));
    IF (htype <> zdata) THEN delay(500)
    END; (* sz_sendbinaryheader *)

  (* 1---------------1 *)

  PROCEDURE sz_sendda32(VAR buf : buftype; blength : integer; frameend : byte);

    VAR
      crc            : longint;
      t              : integer;

    BEGIN (* sz_sendda32 *)
    crc := $ffffffff;
    FOR t := 0 TO (blength - 1) DO BEGIN
      sz_z_sendbyte(buf[t]); crc := updc32(buf[t], crc); END;
    crc := updc32(frameend, crc); crc := (NOT crc);
    z_sendbyte(zdle); z_sendbyte(frameend);
    FOR t := 0 TO 3 DO BEGIN
      sz_z_sendbyte(byte(crc)); crc := (crc shr 8); END;
    z_sendbyte(17); delay(500);
    END; (* sz_sendda32 *)

  (* 1---------------1 *)

  PROCEDURE sz_senddata(VAR buf : buftype; blength : integer; frameend : byte);

    VAR
      crc            : word;
      t              : integer;

    BEGIN (* sz_senddata *)
    IF (usecrc32) THEN BEGIN
      sz_sendda32(buf, blength, frameend); exit; END;     (* <<-- EXIT *)
    crc := 0;
    FOR t := 0 TO (blength - 1) DO BEGIN
      sz_z_sendbyte(buf[t]); crc := updcrc(buf[t], crc); END;
    crc := updcrc(frameend, crc); z_sendbyte(zdle);
    z_sendbyte(frameend);
    crc := updcrc(0, crc); crc := updcrc(0, crc);
    sz_z_sendbyte(lo(crc shr 8)); sz_z_sendbyte(lo(crc));
    IF (frameend = zcrcw) THEN BEGIN
      z_sendbyte(17); delay(500); END
    END; (* sz_senddata *)

  (* 1---------------1 *)

  PROCEDURE sz_endsend;

    VAR
      done           : boolean;

    BEGIN (* sz_endsend *)
    done := false;
    REPEAT
      z_putlongintoheader(txpos);
      sz_sendbinaryheader(zfin, txhdr);
      CASE z_getheader(rxhdr) OF
zfin:   BEGIN
        z_sendbyte(ord('O')); z_sendbyte(ord('O'));
        delay(500); z_clearoutbound; exit;                (* <<-- EXIT *)
        END;
zcan,
rcdo,
zferr,
ztimeout:
        exit;                                             (* <<-- EXIT *)
        END; {case}
    UNTIL (done)
    END; (* sz_endsend *)

  (* 1---------------1 *)

  FUNCTION sz_getreceiverinfo : integer;

    VAR
      rxflags, n, c  : integer;

    BEGIN (* sz_getreceiverinfo *)
    z_message('Getting info.');
    FOR n := 1 TO 10 DO BEGIN
      c := z_getheader(rxhdr); z_frame(c);
      CASE c OF
zchallenge:
        BEGIN
        z_putlongintoheader(rxpos);
        z_sendhexheader(zack, txhdr);
        END;
zcommand:
        BEGIN
        z_putlongintoheader(longint(0));
        z_sendhexheader(zrqinit, txhdr);
        END;
zrinit: BEGIN
        rxbuflen := (word(rxhdr[zp1]) shl 8) OR rxhdr[zp0];
        usecrc32 := ((rxhdr[zf0] AND canfc32) <> 0);
        z_showcheck(usecrc32);
        sz_getreceiverinfo := zok; exit;                  (* <<-- EXIT *)
        END;
zcan,
rcdo,
ztimeout:
        BEGIN
        sz_getreceiverinfo := zerror; exit;               (* <<-- EXIT *)
        END;
      ELSE IF (c <> zrqinit) OR (rxhdr[zf0] <> zcommand) THEN
        z_sendhexheader(znak, txhdr);
        END  {case}
      END; {for}
    sz_getreceiverinfo := zerror;
    END; (* sz_getreceiverinfo *)

  (* 1---------------1 *)

  FUNCTION sz_syncwithreceiver : integer;

    VAR
      c, num_errs    : integer;
      done           : boolean;

    BEGIN (* sz_syncwithreceiver *)
    num_errs := 7; done := false;
    REPEAT
      c := z_getheader(rxhdr); z_frame(c); z_clearinbound;
      CASE c OF
ztimeout:
        BEGIN
        dec(num_errs);
        IF (num_errs < 0) THEN BEGIN
          sz_syncwithreceiver := zerror; exit; END;       (* <<-- EXIT *)
        END;
zcan,
zabort,
zfin,
rcdo:   BEGIN
        sz_syncwithreceiver := zerror; exit;              (* <<-- EXIT *)
        END;
zrpos:  BEGIN
        IF (NOT z_seekfile(infile, rxpos)) THEN BEGIN
          z_message('File seek error');
          sz_syncwithreceiver := zerror; exit; END;       (* <<-- EXIT *)
        z_message('Repositioning...'); z_showloc(rxpos);
        txpos := rxpos; sz_syncwithreceiver := c; exit;   (* <<-- EXIT *)
        END;
zskip,
zrinit,
zack:   BEGIN
        sz_syncwithreceiver := c; exit;                   (* <<-- EXIT *)
        END;
      ELSE BEGIN
        z_message('I dunno what happened!');
        sz_sendbinaryheader(znak, txhdr); END;
        END;  {case}
    UNTIL (done);
    END; (* sz_syncwithreceiver *)

  (* 1---------------1 *)

  FUNCTION sz_sendfiledata : integer;

    LABEL 70 {waitack}, 71 {somemore}, 72 {oops};

    VAR
      c, e           : integer;
      newcnt,
      blklen, blkred,
      maxblklen,
      goodblks,
      goodneeded      : word;

    BEGIN (* sz_sendfiledata *)
    z_message('Sending file...'); goodneeded := 1;
    IF (zbaud < 300) THEN maxblklen := 128
    ELSE maxblklen := (word(zbaud) DIV 300) * 256;
    IF (maxblklen > zbufsize) THEN maxblklen := zbufsize;
    IF (rxbuflen > 0) AND (rxbuflen < maxblklen) THEN
      maxblklen := rxbuflen;
    blklen := maxblklen; ztime := z_settimer;
71: {SOMEMORE}
    IF (z_charavail) THEN BEGIN
70:   {WAITACK}
      c := sz_syncwithreceiver; z_frame(c);
      CASE c OF
zskip:  BEGIN
        sz_sendfiledata := zskip; exit;                   (* <<-- EXIT *)
        END;
zack:   {null};
zrpos:  BEGIN
        inc(zerrors); z_errors(zerrors);
        IF ((blklen shr 2) > 32) THEN
          blklen := (blklen shr 2)
        ELSE blklen := 32;
        goodblks := 0;
        goodneeded := (goodneeded shl 1) OR 1;
        END;
zrinit: BEGIN
        sz_sendfiledata := zok; exit;                     (* <<-- EXIT *)
        END;
      ELSE BEGIN
        sz_sendfiledata := zerror; exit; END;             (* <<-- EXIT *)
        END; (* case *)
      WHILE (z_charavail) DO  BEGIN
        CASE (z_getbyte(1)) OF
can,
zpad:     GOTO 70;                                        (* <<-- WAITACK *)
rcdo:     BEGIN
          sz_sendfiledata := zerror; exit;                (* <<_ EXIT *)
          END
          END;  {case}
        END;
      END; {if char avail}
    newcnt := rxbuflen; z_putlongintoheader(txpos);
    sz_sendbinaryheader(zdata, txhdr);
    z_message('Sending data header');
    REPEAT
      IF (keypressed) THEN
        IF (readkey = #27) THEN BEGIN
          z_message('Aborted from keyboard'); z_sendcan;
          GOTO 72; END;                                   (* <<-- OOPS *)
      IF (NOT z_carrier) THEN GOTO 72;                    (* <<-- OOPS *)
      IF (NOT z_readfile(infile, txbuf, blklen, blkred)) THEN BEGIN
        z_message('Error reading disk'); z_sendcan;
        GOTO 72; END;                                     (* <<-- OOPS *)
      IF (blkred < blklen) THEN e := zcrce
      ELSE IF (rxbuflen <> 0) AND ((newcnt - blkred) <= 0) THEN BEGIN
        newcnt := (newcnt - blkred); e := zcrcw; END
      ELSE e := zcrcg;
      sz_senddata(txbuf, blkred, e);
      txpos := txpos + blkred; z_showloc(txpos);
      inc(goodblks);
      IF (blklen < maxblklen) AND (goodblks > goodneeded) THEN BEGIN
        IF ((blklen shl 1) < maxblklen) THEN
          blklen := (blklen shl 1)
        ELSE blklen := maxblklen;
        goodblks := 0; END;
      IF (e = zcrcw) THEN GOTO 70;                        (* <<-- WAITACK *)
      WHILE (z_charavail) DO BEGIN
        CASE z_getbyte(1) OF
can,
zpad:     BEGIN
          z_message('Trouble?'); z_clearoutbound;
          sz_senddata(txbuf, 0, zcrce); GOTO 70;          (* <<-- WAITACK *)
          END;
rcdo:     BEGIN
          sz_sendfiledata := zerror; exit;                (* <<_- EXIT *)
          END;
          END; {case}
        END; {while}
    UNTIL (e <> zcrcg);
    REPEAT
      z_putlongintoheader(txpos);
      z_message('Sending EOF');
      sz_sendbinaryheader(zeof, txhdr);
      c := sz_syncwithreceiver;
      CASE c OF
zack:   {null};
zrpos:  GOTO 71;                                          (* <<-- SOMEMORE *)
zrinit: BEGIN
        sz_sendfiledata := zok; exit;                     (* <<-- EXIT *)
        END;
zskip:  BEGIN
        sz_sendfiledata := c; exit;                       (* <<-- EXIT *)
        END
      ELSE
72:     {OOPS}
        BEGIN
        sz_sendfiledata := zerror; exit; END;             (* <<-- EXIT *)
        END; (* case *)
    UNTIL (c <> zack);
    END; (* sz_sendfiledata *)

  (* 1---------------1 *)

  FUNCTION sz_sendfile : integer;

    VAR
      c              : integer;
      done           : boolean;

    BEGIN (* sz_sendfile *)
    zerrors := word(0); done := false;
    REPEAT
      IF (keypressed) THEN
        IF (readkey = #27) THEN BEGIN
          z_sendcan; z_message('Aborted from keyboard');
          sz_sendfile := zerror; exit; END;               (* <<-- EXIT *)
      IF (NOT z_carrier) THEN BEGIN
        z_message('Lost carrier');
        sz_sendfile := zerror; exit; END;                 (* <<-- EXIT *)
      fillchar(txhdr, 4, 0); txhdr[zf0] := zcresum; (* recover *)
      sz_sendbinaryheader(zfile, txhdr);
      sz_senddata(txbuf, zbufsize, zcrcw);
      REPEAT
        c := z_getheader(rxhdr); z_frame(c);
        CASE c OF
zcan,
rcdo,
ztimeout,
zfin,
zabort:   BEGIN
          sz_sendfile := zerror; exit;                    (* <<-- EXIT *)
          END;
zrinit:   {null - this will cause a loopback};
zcrc:     BEGIN
          z_putlongintoheader(z_filecrc32(infile));
          z_sendhexheader(zcrc, txhdr);
          END;
zskip:    BEGIN
          sz_sendfile := c; exit;                         (* <<-- EXIT *)
          END;
zrpos:    BEGIN
          IF (NOT z_seekfile(infile, rxpos)) THEN BEGIN
            z_message('File positioning error');
            z_sendhexheader(zferr, txhdr);
            sz_sendfile := zerror; exit; END;             (* <<-- EXIT *)
          z_message('Setting start position');
          z_showloc(rxpos); strtpos := rxpos;
          txpos := rxpos; sz_sendfile := sz_sendfiledata;
          exit;                                           (* <<-- EXIT *)
          END;
          END; {case}
      UNTIL (c <> zrinit);
    UNTIL (done);
    END; (* sz_sendfile *)

  (* 1---------------1 *)

  FUNCTION zmodem_send(pathname : string; lastfile : boolean; comport : word;
                     baudrate : longint) : boolean;
    VAR
      s            : string;
      n            : integer;

    BEGIN (* zmodem_send *)
    zerrors := 0; zbaud := baudrate; zport := comport;
    z_openwindow(tpzver);
    IF (NOT z_asyncon(comport, baudrate)) THEN BEGIN
      z_message('Unable to open port'); delay(2000);
      z_closewindow; zmodem_send := false; exit; END;     (* <<-- EXIT *)
    IF (NOT z_carrier) THEN BEGIN
      z_message('Lost carrier'); delay(2000);
      z_closewindow; z_asyncoff; zmodem_send := false;
      exit; END;                                          (* <<-- EXIT *)
    IF (NOT z_findfile(pathname, fname, fsize, ftime)) THEN BEGIN
      z_message('Unable to find/open file'); sz_endsend;
      z_closewindow; z_asyncoff; zmodem_send := false;
      exit; END;                                          (* <<-- EXIT *)
    z_showname(fname); z_showsize(fsize);
    z_showtransfertime(fsize, zbaud); str(fsize, s);
    s := (fname + #0 + s + ' ');
    s := s + z_tounixdate(ftime); n := length(s);
    FOR n := 1 TO length(s) DO BEGIN
      IF (s[n] IN ['A'..'Z']) THEN s[n] := chr(ord(s[n]) + $20); END;
    fillchar(txbuf, zbufsize, 0);
    move(s[1], txbuf[0], length(s));
    IF (zbaud > 0) THEN
      rxtimeout := integer(614400 DIV zbaud)
    ELSE rxtimeout := 100;
    IF (rxtimeout < 100) THEN rxtimeout := 100;
    attn[0] := ord('r'); attn[1] := ord('z');
    attn[3] := 13; attn[4] := 0; z_putstring(attn);
    fillchar(attn, sizeof(attn), 0);
    z_putlongintoheader(longint(0));
    z_message('Sending ZRQINIT');
    z_sendhexheader(zrqinit, txhdr);
    IF (sz_getreceiverinfo = zerror) THEN BEGIN
      z_closewindow; z_asyncoff; zmodem_send := false;
      exit; END;                                          (* <<-- EXIT *)
    IF (NOT z_openfile(infile, pathname)) THEN
      IF (ioresult <> 0) THEN BEGIN
        z_message('Failure to open file'); z_sendcan;
        z_closewindow; z_asyncoff;
        zmodem_send := false; exit; END;                  (* <<-- EXIT *)
    n := sz_sendfile;
    zcps := (fsize DIV (z_settimer - ztime));
    z_closefile(infile); z_frame(n); str(zcps : 4, s);
    z_message(s+' cps');
    IF (n = zok) AND (lastfile) THEN sz_endsend
    ELSE z_sendcan;
    z_closewindow; z_asyncoff; zmodem_send := true;
    END; (* zmodem_send *)

  (* 1---------------1 *)

  END.  (* tpz *)
`