<?php
/*
 * ----------------------------------------------------------------------------
 * Va-Channels.inc.php                                               11/24/2006
 * This script will retrieve all of the informaton related to the configured
 * channels. The channels.cfg file is use ONLY for backward compatibility with
 * VBBS utilities. You should use the channelx.cfg for VADV.
 * ----------------------------------------------------------------------------
 * Copyright (C) 2002-2006 Steve Winn. All Rights Reserved.
 *
 * The source code contains a fair amount of comments, which may aid anyone
 * attempting to follow the program or to modify it.
 *
 * For licensing and copyright information, view LICENSE.TXT.
 * ----------------------------------------------------------------------------
 */

require_once('include_dir.php');                    // Get include directory.
require_once($includeDir . 'va-functions.inc.php'); // A VADV-PHP script.


/**
* Va-Channels.inc.php :: GetVBBSChannels()
*
* Retrieve all of the channel information and store it into a
* multidimensional array.
*
* Array format: $channels[1]['setting']
*               $channels[0] = Total number of nodes.
*
* @return array
*/
function GetVBBSChannels() {
    $fileString = ReadVBBSChannels();
    if ($fileString === FALSE) {
        trigger_error(VA_Error(3));
    }

    $channels = array();                    // Create the array.
    $offset   = 0;

    for ($x = 1; $offset < strlen($fileString); $x++) {
        $channels[$x]['init']       = GetString($fileString, 1 + $offset, 40);
        $channels[$x]['baudrate']   = GetLongInt($fileString, 41 + $offset);
        $channels[$x]['lockedbaud'] = GetInteger($fileString, 45 + $offset);
        $channels[$x]['rtscts']     = GetInteger($fileString, 47 + $offset);
        $channels[$x]['serialport'] = GetInteger($fileString, 49 + $offset);
        $channels[$x]['porttype']   = GetInteger($fileString, 51 + $offset);
        $channels[$x]['baseaddy']   = dechex(GetInteger($fileString, 53 + $offset));
        $channels[$x]['irq']        = dechex(GetInteger($fileString, 55 + $offset));
        $channels[$x]['minbaud']    = GetLongInt($fileString, 57 + $offset);
        $channels[$x]['extra']      = GetString($fileString, 61 + $offset, 4); 

        $offset = 64 * $x;
    }

    $channels[0] = $x - 1;                  // Put the number of channels in position 0.

    return $channels;                       // Return the channel information.

}


/**
* Va-Channels.inc.php :: PutVBBSChannels()
*
* This function will change the channels.cfg to match the configuration variable
* passed. It then returns the new file string. The configuration variable MUST
* be in the format of $variable[1]['setting'].
*
* @param string $fileString
* @param array $configVariable
* @return string
*/
function PutVBBSChannels($fileString, $configVariable) {
    if (!is_string($fileString)) {
        $fileString = (string) $fileString;
    }
    if (!is_array($configVariable)) {
        trigger_error(VA_Error(1));
    }

    $length  = count($configVariable);      // Get the number of elements in the config variable.
    $pointer = 0;                           // Reset the pointer.
    $offset  = 0;

    if (isset($configVariable[0])) {
        $length--;
    }

    for ($x = 1; $pointer < $length; $x++) {
        $total = intval(strlen($fileString) / 64);
        if ($x > $total) {
            $fileString = $fileString . str_pad('', 64, chr(0));
        }

        if (isset($configVariable[$x])) {
            $pointer++;
            if (isset($configVariable[$x]['init']))       $fileString = PutString($fileString, $configVariable[$x]['init'], 1 + $offset, 40);
            if (isset($configVariable[$x]['baudrate']))   $fileString = PutLongInt($fileString, $configVariable[$x]['baudrate'], 41 + $offset);
            if (isset($configVariable[$x]['lockedbaud'])) $fileString = PutInteger($fileString, $configVariable[$x]['lockedbaud'], 45 + $offset);
            if (isset($configVariable[$x]['rtscts']))     $fileString = PutInteger($fileString, $configVariable[$x]['rtscts'], 47 + $offset);
            if (isset($configVariable[$x]['serialport'])) $fileString = PutInteger($fileString, $configVariable[$x]['serialport'], 49 + $offset);
            if (isset($configVariable[$x]['porttype']))   $fileString = PutInteger($fileString, $configVariable[$x]['porttype'], 51 + $offset);
            if (isset($configVariable[$x]['baseaddy']))   $fileString = PutInteger($fileString, hexdec($configVariable[$x]['baseaddy']), 53 + $offset);
            if (isset($configVariable[$x]['irq']))        $fileString = PutInteger($fileString, hexdec($configVariable[$x]['irq']), 55 + $offset);
            if (isset($configVariable[$x]['minbaud']))    $fileString = PutLongInt($fileString, $configVariable[$x]['minbaud'], 57 + $offset);
            if (isset($configVariable[$x]['extra']))      $fileString = PutString($fileString, $configVariable[$x]['extra'], 61 + $offset, 4);
        }
        $offset = 64 * $x;
    }

    return $fileString;                     // Return the new file string.
}


/**
* Va-Channels.inc.php :: ReadVBBSChannels()
*
* This function will read the channels.cfg file.
*
* @return string
*/
function ReadVBBSChannels() {
    global $VA_SYSPATHS;                    // Include the global variable.

    // Set the filename.
    $file = $VA_SYSPATHS['main'] . '/CHANNELS.CFG';
    $fileString = ReadCfgFile($file);       // Read the file.

    return $fileString;                     // Return the file string.
}


/**
* Va-Channels.inc.php :: WriteVBBSChannels()
*
* This function will write a new channels.cfg file based on the file string
* passed to it. If the file string is not in the correct format, then your
* channels.cfg file will become corrupt and the BBS could stop functioning!
*
* @param string $fileString
* @return mixed
*/
function WriteVBBSChannels($fileString) {
    if (!is_string($fileString)) {
        $fileString = (string) $fileString;
    }

    global $VA_SYSPATHS;                    // Include the global variable.

    // Set the filename.
    $file = $VA_SYSPATHS['main'] . '/CHANNELS.CFG';
    $status = WriteCfgFile($file, $fileString);

    return $status;                         // Return the result status.
}


// Retrieve the channels automatically.
$VA_VBBSCHANNELS = GetVBBSChannels();

?>