<?php
/*
 * ----------------------------------------------------------------------------
 * Va-ChannelX.inc.php                                               11/24/2006
 * This script will retrieve all of the informaton related to the configured
 * channels.
 * ----------------------------------------------------------------------------
 * Copyright (C) 2002-2006 Steve Winn. All Rights Reserved.
 *
 * The source code contains a fair amount of comments, which may aid anyone
 * attempting to follow the program or to modify it.
 *
 * For licensing and copyright information, view LICENSE.TXT.
 * ----------------------------------------------------------------------------
 */

require_once('include_dir.php');                    // Get include directory.
require_once($includeDir . 'va-functions.inc.php'); // A VADV-PHP script.


/**
* Va-ChannelX.inc.php :: GetChannels()
*
* Retrieve all of the channel information and store it into a
* multidimensional array.
*
* Array format: $channels[1]['setting']
*               $channels[0] = Total number of nodes.
*
* @return array
*/
function GetChannels() {
    $fileString = ReadChannels();
    if ($fileString === FALSE) {
        trigger_error(VA_Error(3));
    }

    $channels = array();                    // Create the array.
    $offset   = 0;

    for ($x = 1; $offset < strlen($fileString); $x++) {
        $channels[$x]['init1']      = GetString($fileString, 1 + $offset, 40);
        $channels[$x]['init2']      = GetString($fileString, 41 + $offset, 40);
        $channels[$x]['init3']      = GetString($fileString, 81 + $offset, 40);
        $channels[$x]['answer']     = GetString($fileString, 121 + $offset, 20);
        $channels[$x]['dial']       = GetString($fileString, 141 + $offset, 20);
        $channels[$x]['offhook']    = GetString($fileString, 161 + $offset, 20);
        $channels[$x]['baudrate']   = GetLongInt($fileString, 181 + $offset);
        $channels[$x]['lockedbaud'] = GetInteger($fileString, 185 + $offset);
        $channels[$x]['rtscts']     = GetInteger($fileString, 187 + $offset);
        $channels[$x]['serialport'] = GetInteger($fileString, 189 + $offset);
        $channels[$x]['porttype']   = GetInteger($fileString, 191 + $offset);
        $channels[$x]['baseaddy']   = dechex(GetInteger($fileString, 193 + $offset));
        $channels[$x]['irq']        = dechex(GetInteger($fileString, 195 + $offset));
        $channels[$x]['minbaud']    = GetLongInt($fileString, 197 + $offset);
        $channels[$x]['extra']      = GetString($fileString, 199 + $offset, 4);

        $offset = 204 * $x;
    }

    $channels[0] = $x - 1;                  // Put the number of channels in position 0.

    return $channels;                       // Return the channel information.

}


/**
* Va-ChannelX.inc.php :: PutChannels()
*
* This function will change the channelx.cfg to match the configuration variable
* passed. It then returns the new file string. The configuration variable MUST
* be in the format of $variable[1]['setting'].
*
* @param string $fileString
* @param array $configVariable
* @return string
*/
function PutChannels($fileString, $configVariable) {
    if (!is_string($fileString)) {
        $fileString = (string) $fileString;
    }
    if (!is_array($configVariable)) {
        trigger_error(VA_Error(1));
    }

    $length  = count($configVariable);      // Get the number of elements in the config variable.
    $pointer = 0;                           // Reset the pointer.
    $offset  = 0;

    if (isset($configVariable[0])) {
        $length--;
    }

    for ($x = 1; $pointer < $length; $x++) {
        $total = intval(strlen($fileString) / 204);
        if ($x > $total) {
            $fileString = $fileString . str_pad('', 204, chr(0));
        }

        if (isset($configVariable[$x])) {
            $pointer++;
            if (isset($configVariable[$x]['init1']))      $fileString = PutString($fileString, $configVariable[$x]['init1'], 1 + $offset, 40);
            if (isset($configVariable[$x]['init2']))      $fileString = PutString($fileString, $configVariable[$x]['init2'], 41 + $offset, 40);
            if (isset($configVariable[$x]['init3']))      $fileString = PutString($fileString, $configVariable[$x]['init3'], 81 + $offset, 40);
            if (isset($configVariable[$x]['answer']))     $fileString = PutString($fileString, $configVariable[$x]['answer'], 121 + $offset, 20);
            if (isset($configVariable[$x]['dial']))       $fileString = PutString($fileString, $configVariable[$x]['dial'], 141 + $offset, 20);
            if (isset($configVariable[$x]['offhook']))    $fileString = PutString($fileString, $configVariable[$x]['offhook'], 161 + $offset, 20);
            if (isset($configVariable[$x]['baudrate']))   $fileString = PutLongInt($fileString, $configVariable[$x]['baudrate'], 181 + $offset);
            if (isset($configVariable[$x]['lockedbaud'])) $fileString = PutInteger($fileString, $configVariable[$x]['lockedbaud'], 185 + $offset);
            if (isset($configVariable[$x]['rtscts']))     $fileString = PutInteger($fileString, $configVariable[$x]['rtscts'], 187 + $offset);
            if (isset($configVariable[$x]['serialport'])) $fileString = PutInteger($fileString, $configVariable[$x]['serialport'], 189 + $offset);
            if (isset($configVariable[$x]['porttype']))   $fileString = PutInteger($fileString, $configVariable[$x]['porttype'], 191 + $offset);
            if (isset($configVariable[$x]['baseaddy']))   $fileString = PutInteger($fileString, hexdec($configVariable[$x]['baseaddy']), 193 + $offset);
            if (isset($configVariable[$x]['irq']))        $fileString = PutInteger($fileString, hexdec($configVariable[$x]['irq']), 195 + $offset);
            if (isset($configVariable[$x]['minbaud']))    $fileString = PutLongInt($fileString, $configVariable[$x]['minbaud'], 197 + $offset);
            if (isset($configVariable[$x]['extra']))      $fileString = PutString($fileString, $configVariable[$x]['extra'], 199 + $offset, 4);
        }
        $offset = 204 * $x;
    }

    return $fileString;                     // Return the new file string.
}


/**
* Va-ChannelX.inc.php :: ReadChannels()
*
* This function will read the channelx.cfg file.
*
* @return string
*/
function ReadChannels() {
    global $VA_SYSPATHS;                    // Include the global variable.

    // Set the filename.
    $file = $VA_SYSPATHS['main'] . '/CHANNELX.CFG';
    $fileString = ReadCfgFile($file);       // Read the file.

    return $fileString;                     // Return the file string.
}


/**
* Va-ChannelX.inc.php :: WriteChannels()
*
* This function will write a new channelx.cfg file based on the file string
* passed to it. If the file string is not in the correct format, then your
* channelx.cfg file will become corrupt and the BBS could stop functioning!
*
* @param string $fileString
* @return mixed
*/
function WriteChannels($fileString) {
    if (!is_string($fileString)) {
        $fileString = (string) $fileString;
    }

    global $VA_SYSPATHS;                    // Include the global variable.

    // Set the filename.
    $file = $VA_SYSPATHS['main'] . '/CHANNELX.CFG';
    $status = WriteCfgFile($file, $fileString);

    return $status;                         // Return the result status.
}


// Retrieve the channels automatically.
$VA_CHANNELS = GetChannels();

?>