<?php
/*
 * ----------------------------------------------------------------------------
 * Va-Functions.inc.php                                              01/03/2007
 * This script contains a set of functions that will make it easier to get
 * information from the Virtual Advanced configuration and data files. It also
 * contains basic functions to aid you when creating new scripts.
 *
 * This script should always be included first in any script that will use any
 * VADV-PHP functions.
 * ----------------------------------------------------------------------------
 * Copyright (C) 2002-2006 Steve Winn. All Rights Reserved.
 *
 * The source code contains a fair amount of comments, which may aid anyone
 * attempting to follow the program or to modify it.
 *
 * For licensing and copyright information, view LICENSE.TXT.
 * ----------------------------------------------------------------------------
 */

$VERSION = '1.0.070103';                            // Current VADV-PHP version. DO NOT CHANGE.

/*
 * The DEBUG variables can be altered so you can receive any error messages that
 * occur while in operation. By default, this feature is turned off.
 *
 * Note that when customizing the information below for your account, this
 * feature is known to NOT work with some mail configurations.
 */
$DEBUG_MAIL = FALSE;                                // Turn on/off debug mailer.
                                                    // This should only be used while developing
                                                    // and turned off on a production site.

$DEBUG_EMAIL = 'bugs@vadvphp.com';                  // The email address to send bug reports to.

$DEBUG_SMTP_SERVER = 'mail.at2k.org';               // The SMTP server address to use to send
                                                    // any bug reports out of.

//set_error_handler('CustomErrorHandler');          // Set the default error handler.
error_reporting(E_ERROR |                           // Set errors to be reported,
                E_WARNING |                         // which is used by the error handler.
                E_NOTICE |
                E_USER_ERROR |
                E_USER_WARNING |
                E_USER_NOTICE);

require_once('include_dir.php');                    // Get include directory.
require_once($includeDir . 'common.inc.php');       // Include common settings.
require_once($includeDir . 'va-syspaths.inc.php');  // Include the BBS's paths.
require_once($includeDir . 'va-loginauth.inc.php'); // Login authentication.
require_once($includeDir . 'va-stats.inc.php');     // Include BBS statistics.
require_once($includeDir . 'vadv-php_dat.inc.php'); // Include User settings.

$USER_SETTINGS = GetUserSettings($USER['userno']);

srand((double) microtime() * 1000000);              // Seed random number generator.
if (!isset($VA_ACTION)) {
    $VA_ACTION = 'Browsing';
}
$VA_USERS = UpdateWhosOnline($VA_ACTION, $TIMEOUT); // Update the WEBCALLER.LST file.

// Increase the hit counter by one if a new session.
if (((!isset($_SESSION['hitcounter'])) || (!$_SESSION['hitcounter'])) &&
    (!DetectSpider())) {
    $fileString = ReadStats();
    $stats = GetStats($fileString);
    $stats['webhitstotal']++;
    $fileString = PutStats($fileString, $stats);
    WriteStats($fileString);
    $_SESSION['hitcounter'] = TRUE;
}


// --- BEGIN FUNCTIONS ---


/**
* Va-Functions.inc.php :: Array_StripSlashes()
*
* This function will remove slashes from an array.
*
* @param array $array
* @return array
*/
function Array_StripSlashes($array) {
    if (!is_array($array)) {
        return $array;
    }
    while (list($key) = each($array)) {
        if (is_array($array[$key])) {
            array_stripslashes($array[$key]);
        } else {
            $array[$key] = stripslashes($array[$key]);
        }
    }
    
    return $array;
}


/**
* Va-Functions.inc.php :: ConvertDateTime()
*
* This function will convert the many formats of dates and times into a
* standard format that VADV-PHP will use. You can specify your own formats
* if you wish by using the same parameters that date() uses.
*
* @param string $string
* @return array
*/
function ConvertDateTime($string, $dateFormat = '', $timeFormat = '') {
    if (!is_string($string)) {
        $string = (string) $string;
    }
    if (!is_string($dateFormat)) {
        $dateFormat = (string) $dateFormat;
    }
    if (!is_string($timeFormat)) {
        $timeFormat = (string) $timeFormat;
    }

    global $USER_SETTINGS;

    $date['date'] = 'Date';
    $date['time'] = 'Time';
    $date['tz'] = TRUE;

    $string = trim($string);
    if ($string == '') {
        return $date;
    }

    $string = str_replace('(', '', $string);
    $string = str_replace(')', '', $string);
    $string = str_replace('[', '', $string);
    $string = str_replace(']', '', $string);

    if ((((strpos($string, '-12'))  || (strpos($string, '-1200'))) !== FALSE) ||
        (((strpos($string, '-11'))  || (strpos($string, '-1100'))) !== FALSE) ||
        (((strpos($string, '-10'))  || (strpos($string, '-1000'))) !== FALSE) ||
        (((strpos($string, '-9'))   || (strpos($string, '-900'))  || (strpos($string, '-0900'))) !== FALSE) ||
        (((strpos($string, '-930')) || (strpos($string, '-0930'))) !== FALSE) ||
        (((strpos($string, '-8'))   || (strpos($string, '-800'))  || (strpos($string, '-0800'))) !== FALSE) ||
        (((strpos($string, '-7'))   || (strpos($string, '-700'))  || (strpos($string, '-0700'))) !== FALSE) ||
        (((strpos($string, '-6'))   || (strpos($string, '-600'))  || (strpos($string, '-0600'))) !== FALSE) ||
        (((strpos($string, '-5'))   || (strpos($string, '-500'))  || (strpos($string, '-0500'))) !== FALSE) ||
        (((strpos($string, '-4'))   || (strpos($string, '-400'))  || (strpos($string, '-0400'))) !== FALSE) ||
        (((strpos($string, '-330')) || (strpos($string, '-0330'))) !== FALSE) ||
        (((strpos($string, '-3'))   || (strpos($string, '-300'))  || (strpos($string, '-0300'))) !== FALSE) ||
        (((strpos($string, '-2'))   || (strpos($string, '-200'))  || (strpos($string, '-0200'))) !== FALSE) ||
        (((strpos($string, '-1'))   || (strpos($string, '-100'))  || (strpos($string, '-0100'))) !== FALSE) ||
        ((strpos($string, '+0000')) !== FALSE) ||
        (((strpos($string, '+1'))   || (strpos($string, '+100'))  || (strpos($string, '+0100'))) !== FALSE) ||
        (((strpos($string, '+2'))   || (strpos($string, '+200'))  || (strpos($string, '+0200'))) !== FALSE) ||
        (((strpos($string, '+3'))   || (strpos($string, '+300'))  || (strpos($string, '+0300'))) !== FALSE) ||
        (((strpos($string, '+330')) || (strpos($string, '+0330'))) !== FALSE) ||
        (((strpos($string, '+4'))   || (strpos($string, '+400'))  || (strpos($string, '+0400'))) !== FALSE) ||
        (((strpos($string, '+430')) || (strpos($string, '+0430'))) !== FALSE) ||
        (((strpos($string, '+5'))   || (strpos($string, '+500'))  || (strpos($string, '+0500'))) !== FALSE) ||
        (((strpos($string, '+530')) || (strpos($string, '+0530'))) !== FALSE) ||
        (((strpos($string, '+545')) || (strpos($string, '+0545'))) !== FALSE) ||
        (((strpos($string, '+6'))   || (strpos($string, '+600'))  || (strpos($string, '+0600'))) !== FALSE) ||
        (((strpos($string, '+630')) || (strpos($string, '+0630'))) !== FALSE) ||
        (((strpos($string, '+7'))   || (strpos($string, '+700'))  || (strpos($string, '+0700'))) !== FALSE) ||
        (((strpos($string, '+8'))   || (strpos($string, '+800'))  || (strpos($string, '+0800'))) !== FALSE) ||
        (((strpos($string, '+845')) || (strpos($string, '+0845'))) !== FALSE) ||
        (((strpos($string, '+9'))   || (strpos($string, '+900'))  || (strpos($string, '+0900'))) !== FALSE) ||
        (((strpos($string, '+930')) || (strpos($string, '+0930'))) !== FALSE) ||
        (((strpos($string, '+10'))  || (strpos($string, '+1000'))) !== FALSE) ||
        (((strpos($string, '+1030'))) !== FALSE) ||
        (((strpos($string, '+11'))  || (strpos($string, '+1100'))) !== FALSE) ||
        (((strpos($string, '+1130'))) !== FALSE) ||
        (((strpos($string, '+12'))  || (strpos($string, '+1200'))) !== FALSE) ||
        (((strpos($string, '+1245'))) !== FALSE) ||
        (((strpos($string, '+13'))  || (strpos($string, '+1300'))) !== FALSE) ||
        (((strpos($string, '+14'))  || (strpos($string, '+1400'))) !== FALSE)) {
        $date['tz'] = TRUE;
    } else {
        $date['tz'] = FALSE;
    }

    $tzString = "/\bacdt|acst|adt|aedt|aest|akdt|akst|ast|awst|bst|cdt|cest|cet|cst|cxt|edt|eest|eet|est|gmt|haa|hac|hadt|hae|hap|har|hast|hat|hay|hna|hnc|hne|hnp|hnr|hnt|hny|ist|mdt|mesz|mez|mst|ndt|nft|nst|pdt|pst|utc|west|wet|wst\b/i";
    if ($date['tz']) {                              // If it found tz information, then strip any tz strings.
        $string = preg_replace($tzString, '', $string);
    } else if (preg_match($tzString, $string)) {    // Otherwise, if it contains one of the strings, then set tz to true.
        $date['tz'] = TRUE;
    }

    $timeStamp = strtotime($string);

    if ($timeStamp == -1) {
        $pos  = strpos($string, '19');
        $year = (integer) substr($string, $pos, 4);
        while (($year < 1900) && ($pos !== FALSE)) {
            $pos = strpos($string, '19', $pos + 1);
            $year = (integer) substr($string, $pos, 4);
        }
        if (($year < 1970) && ($year >= 1900)) {
            $newstring = substr_replace($string, $year + 100, $pos, 4);
            $timeStamp = strtotime($newstring);
        } else {
            $a = substr($string, 0, 10);
            $b = substr($string, 10, 9);
            $c = substr($string, 19, 5);
            $newstring = $a . $c . $b;
            $timeStamp = strtotime($newstring);
        }
        if ($timeStamp == -1) {
            return $date;
        }
    }

    $timeStamp = $timeStamp - (date("O") / 100 * 60 * 60) + ($USER_SETTINGS['timezone'] / 100 * 60 * 60);

    if (trim($dateFormat) == '') {
        $date['date'] = trim(date("M j, Y", $timeStamp));
    } else {
        $date['date'] = trim(date($dateFormat, $timeStamp));
    }
    if (trim($timeFormat) == '') {
        $date['time'] = trim(date("g:i A", $timeStamp));
    } else {
        $date['time'] = trim(date($timeFormat, $timeStamp));
    }
    
    return $date;
}


/**
* Va-Functions :: ConvertLongFilename()
*
* This function will convert a long filename into a DOS 8.3 filename.
* You can pass the directory the filename will reside in ($dir), and if it
* detects multiple files with the same name it can rename it without disrupting
* the exiting files.
*
* @param string $filename
* @param string $dir
* @return string
*/
function ConvertLongFilename($filename, $dir = '') {
    if (!is_string($filename)) {
        $filename = (string) $filename;
    }
    if (!is_string($dir)) {
        $dir      = (string) $dir;
    }

    $filename = trim($filename);
    $filename = str_replace(' ', '_', $filename);   // Replace any spaces with an underscore.

    $pos = strrpos($filename, '.');         // Find the period.
    if (($pos == 0) || ($pos === FALSE)) {  // If there is no extension, then set
        $pos = strlen($filename);           // it to the end of the file.
    }

    $file = substr($filename, 0, $pos);     // Get the filename.
    if ($pos < strlen($filename)) {
        $ext = substr($filename, $pos + 1); // Get the extension.
        if (strlen($ext) > 3) {
            $ext = substr($ext, 0, 3);
        }
    } else {
        $ext = '';
    }

    // If the filename is 8.3 compliant create the filename variable.
    if (strlen($file) <= 8) {
        $filename = $file . '.' . $ext;
    } else {
        $file = substr($file, 0, 6);
        $filename = $file . '~1.' . $ext;
    }

    if ($dir != '') {
        $path = str_replace('/', "\\", $dir);
        $filepath = $path . '/' . $filename;
        if (is_file($filepath)) {           // If that file already exists...
            $file = substr($file, 0, 6);
            $filename = $file . '~1.' . $ext;
            $filepath = $path . '/' . $filename;
            for ($x = 1; (is_file($filepath) && ($x < 99)); $x++) {
                if ($x == 10) {
                    $file = substr($file, 0, 5);
                }
                $filename = $file . '~' . $x . '.' . $ext;
                $filepath = $path . '/' . $filename;
            }
        }
    }
    
    return $filename;
}


/**
* Va-Functions.inc.php :: CustomErrorHandler()
*
* This is a custom error handler in case all hell breaks loose.
*
* @param integer $errNumber
* @param string $errDesc
* @param string $errFile
* @param integer $errLine
*/
function CustomErrorHandler($errNumber, $errDesc, $errFile, $errLine) {
    if (!is_int($errNumber)) {
        $errNumber = (integer) $errNumber;
    }
    if (!is_string($errDesc)) {
        $errDesc   = (string)  $errDesc;
    }
    if (!is_string($errFile)) {
        $errFile   = (string)  $errFile;
    }
    if (!is_int($errLine)) {
        $errLine   = (integer) $errLine;
    }

    global $VERSION;
    global $DEBUG_MAIL;
    global $DEBUG_EMAIL;
    global $DEBUG_SMTP_SERVER;
    
    // Send an email to the VADV-PHP bug report address.
    if ($DEBUG_MAIL == TRUE) {
        if (gethostbyname($DEBUG_SMTP_SERVER) != $DEBUG_SMTP_SERVER) {
            $phpVersion = phpversion();
            $currentURL = GetCurrentURL();

            $pos  = strrpos($errFile, chr(92)) + 1;
            $file = substr($errFile, $pos);

            // Create error message.
            $error = <<<EOD
Error number: $errNumber
Description:  $errDesc
Script name:  $file
Line number:  $errLine

URL:          $currentURL
PHP version:  $phpVersion
EOD;

            ini_set('SMTP', $DEBUG_SMTP_SERVER);
            ini_set('sendmail_from', $DEBUG_EMAIL);
            $headers  = "Date: " . date("r") . "\r\n";
            $headers .= "Reply-To: " . $DEBUG_EMAIL . "\r\n";
            $headers .= "MIME-Version: 1.0\r\n";
            $headers .= "Content-Type: text/plain; charset=us-ascii\r\n";
            $headers .= "Content-Transfer-Encoding: 7bit\r\n";
            $headers .= "X-Sender: Bug Report<" . $DEBUG_EMAIL . ">\r\n";
            $headers .= "X-Mailer: PHP/" . $phpVersion . "\r\n";
            $headers .= "X-Priority: 3\r\n";
            $headers .= "Return-Path: <" . $DEBUG_EMAIL . ">\r\n";

            mail($DEBUG_EMAIL,
                 '[NON-FATAL] VADV-PHP ERROR (' . $VERSION . ')',
                 'http://' . $_SERVER['SERVER_NAME'] . "\n" . $error,
                 $headers);
            ini_restore('SMTP');
            ini_restore('sendmail_from');
        }
    }

    // Display error message to user.
    print "<h2>Unable to display page.</h2>\n";
    print "<p>An error occurred while processing your request. The site administrator has been notified.</p>\n";
    print "<p>We apologize for the inconvenience.</p>";
    VA_Die('This page will attempt to be redirected in 5 seconds.');
}


/**
* Va-Functions.inc.php :: DetectSpider()
*
* This function detects if the current user client is a known search engine
* spider. If so, it returns TRUE, else it returns FALSE.
*
* @return boolean
*/
function DetectSpider() {
    // The following search engine identifier information was obtained from:
    // http://www.jafsoft.com/searchengines/webbots.html
    if ((isset($_SERVER['HTTP_USER_AGENT'])) &&
        // Webbots...
        ((stristr($_SERVER['HTTP_USER_AGENT'], 'AbachoBOT')                     !== FALSE) ||
         (stristr($_SERVER['HTTP_USER_AGENT'], 'ABCdatos')                      !== FALSE) ||
         (stristr($_SERVER['HTTP_USER_AGENT'], 'AESOP_com_SpiderMan')           !== FALSE) ||
         (stristr($_SERVER['HTTP_USER_AGENT'], 'ah-ha.com')                     !== FALSE) ||
         (stristr($_SERVER['HTTP_USER_AGENT'], 'ia_archiver')                   !== FALSE) ||
         (stristr($_SERVER['HTTP_USER_AGENT'], 'Scooter')                       !== FALSE) ||
         (stristr($_SERVER['HTTP_USER_AGENT'], 'Mercator')                      !== FALSE) ||
         (stristr($_SERVER['HTTP_USER_AGENT'], 'Scooter2_Mercator_3')           !== FALSE) ||
         (stristr($_SERVER['HTTP_USER_AGENT'], 'roach.smo.av.com')              !== FALSE) ||
         (stristr($_SERVER['HTTP_USER_AGENT'], 'Tv<nn>_Merc_resh_26_1_D')       !== FALSE) ||
         (stristr($_SERVER['HTTP_USER_AGENT'], 'AltaVista')                     !== FALSE) ||
         (stristr($_SERVER['HTTP_USER_AGENT'], 'FAST-WebCrawler')               !== FALSE) ||
         (stristr($_SERVER['HTTP_USER_AGENT'], 'Wget')                          !== FALSE) ||
         (stristr($_SERVER['HTTP_USER_AGENT'], 'Acoon Robot')                   !== FALSE) ||
         (stristr($_SERVER['HTTP_USER_AGENT'], 'antibot')                       !== FALSE) ||
         (stristr($_SERVER['HTTP_USER_AGENT'], 'Atomz')                         !== FALSE) ||
         (stristr($_SERVER['HTTP_USER_AGENT'], 'AxmoRobot')                     !== FALSE) ||
         (stristr($_SERVER['HTTP_USER_AGENT'], 'Buscaplus Robi')                !== FALSE) ||
         (stristr($_SERVER['HTTP_USER_AGENT'], 'CanSeek')                       !== FALSE) ||
         (stristr($_SERVER['HTTP_USER_AGENT'], 'ChristCRAWLER')                 !== FALSE) ||
         (stristr($_SERVER['HTTP_USER_AGENT'], 'Clushbot')                      !== FALSE) ||
         (stristr($_SERVER['HTTP_USER_AGENT'], 'Crawler')                       !== FALSE) ||
         (stristr($_SERVER['HTTP_USER_AGENT'], 'DaAdLe.com ROBOT')              !== FALSE) ||
         (stristr($_SERVER['HTTP_USER_AGENT'], 'RaBot')                         !== FALSE) ||
         (stristr($_SERVER['HTTP_USER_AGENT'], 'DeepIndex')                     !== FALSE) ||
         (stristr($_SERVER['HTTP_USER_AGENT'], 'DittoSpyder')                   !== FALSE) ||
         (stristr($_SERVER['HTTP_USER_AGENT'], 'Jack')                          !== FALSE) ||
         (stristr($_SERVER['HTTP_USER_AGENT'], 'Speedy Spider')                 !== FALSE) ||
         (stristr($_SERVER['HTTP_USER_AGENT'], 'ArchitextSpider')               !== FALSE) ||
         (stristr($_SERVER['HTTP_USER_AGENT'], 'ArchitectSpider')               !== FALSE) ||
         (stristr($_SERVER['HTTP_USER_AGENT'], 'EuripBot')                      !== FALSE) ||
         (stristr($_SERVER['HTTP_USER_AGENT'], 'Arachnoidea')                   !== FALSE) ||
         (stristr($_SERVER['HTTP_USER_AGENT'], 'EZResult')                      !== FALSE) ||
         (stristr($_SERVER['HTTP_USER_AGENT'], 'Fast PartnerSite')              !== FALSE) ||
         (stristr($_SERVER['HTTP_USER_AGENT'], 'FAST Data Search')              !== FALSE) ||
         (stristr($_SERVER['HTTP_USER_AGENT'], 'KIT-Fireball')                  !== FALSE) ||
         (stristr($_SERVER['HTTP_USER_AGENT'], 'FyberSearch')                   !== FALSE) ||
         (stristr($_SERVER['HTTP_USER_AGENT'], 'GalaxyBot')                     !== FALSE) ||
         (stristr($_SERVER['HTTP_USER_AGENT'], 'geckobot')                      !== FALSE) ||
         (stristr($_SERVER['HTTP_USER_AGENT'], 'GenCrawler')                    !== FALSE) ||
         (stristr($_SERVER['HTTP_USER_AGENT'], 'GeonaBot')                      !== FALSE) ||
         (stristr($_SERVER['HTTP_USER_AGENT'], 'Googlebot')                     !== FALSE) ||
         (stristr($_SERVER['HTTP_USER_AGENT'], 'moget')                         !== FALSE) ||
         (stristr($_SERVER['HTTP_USER_AGENT'], 'Aranha')                        !== FALSE) ||
         (stristr($_SERVER['HTTP_USER_AGENT'], 'Slurp')                         !== FALSE) ||
         (stristr($_SERVER['HTTP_USER_AGENT'], 'Toutatis')                      !== FALSE) ||
         (stristr($_SERVER['HTTP_USER_AGENT'], 'Hubater')                       !== FALSE) ||
         (stristr($_SERVER['HTTP_USER_AGENT'], 'IlTrovatore-Setaccio')          !== FALSE) ||
         (stristr($_SERVER['HTTP_USER_AGENT'], 'IncyWincy')                     !== FALSE) ||
         (stristr($_SERVER['HTTP_USER_AGENT'], 'UltraSeek')                     !== FALSE) ||
         (stristr($_SERVER['HTTP_USER_AGENT'], 'InfoSeek')                      !== FALSE) ||
         (stristr($_SERVER['HTTP_USER_AGENT'], 'Mole2')                         !== FALSE) ||
         (stristr($_SERVER['HTTP_USER_AGENT'], 'MP3Bot')                        !== FALSE) ||
         (stristr($_SERVER['HTTP_USER_AGENT'], 'ip3000')                        !== FALSE) ||
         (stristr($_SERVER['HTTP_USER_AGENT'], 'kuloko-bot')                    !== FALSE) ||
         (stristr($_SERVER['HTTP_USER_AGENT'], 'LNSpiderguy')                   !== FALSE) ||
         (stristr($_SERVER['HTTP_USER_AGENT'], 'Linknzbot')                     !== FALSE) ||
         (stristr($_SERVER['HTTP_USER_AGENT'], 'lookbot')                       !== FALSE) ||
         (stristr($_SERVER['HTTP_USER_AGENT'], 'MantraAgent')                   !== FALSE) ||
         (stristr($_SERVER['HTTP_USER_AGENT'], 'NetResearchServer')             !== FALSE) ||
         (stristr($_SERVER['HTTP_USER_AGENT'], 'Lycos')                         !== FALSE) ||
         (stristr($_SERVER['HTTP_USER_AGENT'], 'JoocerBot')                     !== FALSE) ||
         (stristr($_SERVER['HTTP_USER_AGENT'], 'HenryTheMiragoRobot')           !== FALSE) ||
         (stristr($_SERVER['HTTP_USER_AGENT'], 'MojeekBot')                     !== FALSE) ||
         (stristr($_SERVER['HTTP_USER_AGENT'], 'mozDex')                        !== FALSE) ||
         (stristr($_SERVER['HTTP_USER_AGENT'], 'MSNBOT')                        !== FALSE) ||
         (stristr($_SERVER['HTTP_USER_AGENT'], 'Navadoo Crawler')               !== FALSE) ||
         (stristr($_SERVER['HTTP_USER_AGENT'], 'Gulliver')                      !== FALSE) ||
         (stristr($_SERVER['HTTP_USER_AGENT'], 'ObjectsSearch')                 !== FALSE) ||
         (stristr($_SERVER['HTTP_USER_AGENT'], 'OnetSzukaj')                    !== FALSE) ||
         (stristr($_SERVER['HTTP_USER_AGENT'], 'PicoSearch')                    !== FALSE) ||
         (stristr($_SERVER['HTTP_USER_AGENT'], 'PJspider')                      !== FALSE) ||
         (stristr($_SERVER['HTTP_USER_AGENT'], 'DIIbot')                        !== FALSE) ||
         (stristr($_SERVER['HTTP_USER_AGENT'], 'nttdirectory_robot')            !== FALSE) ||
         (stristr($_SERVER['HTTP_USER_AGENT'], 'griffon')                       !== FALSE) ||
         (stristr($_SERVER['HTTP_USER_AGENT'], 'maxbot')                        !== FALSE) ||
         (stristr($_SERVER['HTTP_USER_AGENT'], 'gazz')                          !== FALSE) ||
         (stristr($_SERVER['HTTP_USER_AGENT'], 'NationalDirectory-SuperSpider') !== FALSE) ||
         (stristr($_SERVER['HTTP_USER_AGENT'], 'naver')                         !== FALSE) ||
         (stristr($_SERVER['HTTP_USER_AGENT'], 'noxtrumbot')                    !== FALSE) ||
         (stristr($_SERVER['HTTP_USER_AGENT'], 'Openfind')                      !== FALSE) ||
         (stristr($_SERVER['HTTP_USER_AGENT'], 'Openbot')                       !== FALSE) ||
         (stristr($_SERVER['HTTP_USER_AGENT'], 'psbot')                         !== FALSE) ||
         (stristr($_SERVER['HTTP_USER_AGENT'], 'CrawlerBoy')                    !== FALSE) ||
         (stristr($_SERVER['HTTP_USER_AGENT'], 'QweeryBot')                     !== FALSE) ||
         (stristr($_SERVER['HTTP_USER_AGENT'], 'AlkalineBOT')                   !== FALSE) ||
         (stristr($_SERVER['HTTP_USER_AGENT'], 'StackRambler')                  !== FALSE) ||
         (stristr($_SERVER['HTTP_USER_AGENT'], 'SeznamBot')                     !== FALSE) ||
         (stristr($_SERVER['HTTP_USER_AGENT'], 'Search-10')                     !== FALSE) ||
         (stristr($_SERVER['HTTP_USER_AGENT'], 'Fluffy')                        !== FALSE) ||
         (stristr($_SERVER['HTTP_USER_AGENT'], 'Scrubby')                       !== FALSE) ||
         (stristr($_SERVER['HTTP_USER_AGENT'], 'asterias')                      !== FALSE) ||
         (stristr($_SERVER['HTTP_USER_AGENT'], 'speedfind')                     !== FALSE) ||
         (stristr($_SERVER['HTTP_USER_AGENT'], 'Kototoi')                       !== FALSE) ||
         (stristr($_SERVER['HTTP_USER_AGENT'], 'SearchByUsa')                   !== FALSE) ||
         (stristr($_SERVER['HTTP_USER_AGENT'], 'Searchspider')                  !== FALSE) ||
         (stristr($_SERVER['HTTP_USER_AGENT'], 'SightQuestBot')                 !== FALSE) ||
         (stristr($_SERVER['HTTP_USER_AGENT'], 'Spider_Monkey')                 !== FALSE) ||
         (stristr($_SERVER['HTTP_USER_AGENT'], 'Surfnomore')                    !== FALSE) ||
         (stristr($_SERVER['HTTP_USER_AGENT'], 'teoma')                         !== FALSE) ||
         (stristr($_SERVER['HTTP_USER_AGENT'], 'Teradex_Mapper')                !== FALSE) ||
         (stristr($_SERVER['HTTP_USER_AGENT'], 'ESISmartSpider')                !== FALSE) ||
         (stristr($_SERVER['HTTP_USER_AGENT'], 'TraficDublu')                   !== FALSE) ||
         (stristr($_SERVER['HTTP_USER_AGENT'], 'Tutorial Crawler')              !== FALSE) ||
         (stristr($_SERVER['HTTP_USER_AGENT'], 'UK Searcher')                   !== FALSE) ||
         (stristr($_SERVER['HTTP_USER_AGENT'], 'Vivante')                       !== FALSE) ||
         (stristr($_SERVER['HTTP_USER_AGENT'], 'appie')                         !== FALSE) ||
         (stristr($_SERVER['HTTP_USER_AGENT'], 'Nazilla')                       !== FALSE) ||
         (stristr($_SERVER['HTTP_USER_AGENT'], 'WebWombat')                     !== FALSE) ||
         (stristr($_SERVER['HTTP_USER_AGENT'], 'marvin')                        !== FALSE) ||
         (stristr($_SERVER['HTTP_USER_AGENT'], 'MuscatFerret')                  !== FALSE) ||
         (stristr($_SERVER['HTTP_USER_AGENT'], 'WhizBang')                      !== FALSE) ||
         (stristr($_SERVER['HTTP_USER_AGENT'], 'ZyBorg')                        !== FALSE) ||
         (stristr($_SERVER['HTTP_USER_AGENT'], 'WIRE WebRefiner')               !== FALSE) ||
         (stristr($_SERVER['HTTP_USER_AGENT'], 'WSCbot')                        !== FALSE) ||
         (stristr($_SERVER['HTTP_USER_AGENT'], 'Yandex')                        !== FALSE) ||
         (stristr($_SERVER['HTTP_USER_AGENT'], 'Yellopet-Spider')               !== FALSE) ||
         (stristr($_SERVER['HTTP_USER_AGENT'], 'Findexa Crawler')               !== FALSE) ||
         (stristr($_SERVER['HTTP_USER_AGENT'], 'YBSbot')                        !== FALSE) ||
         (stristr($_SERVER['HTTP_USER_AGENT'], 'Iron33')                        !== FALSE) ||
        // Research bots...
         (stristr($_SERVER['HTTP_USER_AGENT'], 'citenikbot')                    !== FALSE) ||
         (stristr($_SERVER['HTTP_USER_AGENT'], 'CLIPS-index')                   !== FALSE) ||
         (stristr($_SERVER['HTTP_USER_AGENT'], 'cosmos')                        !== FALSE) ||
         (stristr($_SERVER['HTTP_USER_AGENT'], 'D2KWebCrawler')                 !== FALSE) ||
         (stristr($_SERVER['HTTP_USER_AGENT'], 'DiaGem')                        !== FALSE) ||
         (stristr($_SERVER['HTTP_USER_AGENT'], 'Digimarc')                      !== FALSE) ||
         (stristr($_SERVER['HTTP_USER_AGENT'], 'EchO!')                         !== FALSE) ||
         (stristr($_SERVER['HTTP_USER_AGENT'], 'FinaleRobot')                   !== FALSE) ||
         (stristr($_SERVER['HTTP_USER_AGENT'], 'Ideare - SignSite')             !== FALSE) ||
         (stristr($_SERVER['HTTP_USER_AGENT'], 'GentleSpider')                  !== FALSE) ||
         (stristr($_SERVER['HTTP_USER_AGENT'], 'Gulper Web Bot')                !== FALSE) ||
         (stristr($_SERVER['HTTP_USER_AGENT'], 'Iarbin')                        !== FALSE) ||
         (stristr($_SERVER['HTTP_USER_AGENT'], 'IRLbot')                        !== FALSE) ||
         (stristr($_SERVER['HTTP_USER_AGENT'], 'KnowItAll')                     !== FALSE) ||
         (stristr($_SERVER['HTTP_USER_AGENT'], 'MJ12bot')                       !== FALSE) ||
         (stristr($_SERVER['HTTP_USER_AGENT'], 'OntoSpider')                    !== FALSE) ||
         (stristr($_SERVER['HTTP_USER_AGENT'], 'sherlock_spider')               !== FALSE) ||
         (stristr($_SERVER['HTTP_USER_AGENT'], 'S.T.A.L.K.E.R.')                !== FALSE) ||
         (stristr($_SERVER['HTTP_USER_AGENT'], 'Steeler')                       !== FALSE) ||
         (stristr($_SERVER['HTTP_USER_AGENT'], 'ru-robot')                      !== FALSE) ||
         (stristr($_SERVER['HTTP_USER_AGENT'], 'USyd-NLP-Spider')               !== FALSE) ||
         (stristr($_SERVER['HTTP_USER_AGENT'], 'WebGather')                     !== FALSE) ||
         (stristr($_SERVER['HTTP_USER_AGENT'], 'xyro')                          !== FALSE) ||
         (stristr($_SERVER['HTTP_USER_AGENT'], 'Zao/')                          !== FALSE) ||
         (stristr($_SERVER['HTTP_USER_AGENT'], 'Zao-Crawler')                   !== FALSE) ||
        // Bots not listed at jafsoft.com...
         (stristr($_SERVER['HTTP_USER_AGENT'], 'Baiduspider')                   !== FALSE) ||
         (stristr($_SERVER['HTTP_USER_AGENT'], 'BecomeBot')                     !== FALSE) ||
         (stristr($_SERVER['HTTP_USER_AGENT'], 'Exabot')                        !== FALSE) ||
         (stristr($_SERVER['HTTP_USER_AGENT'], 'Gigaroo')                       !== FALSE) ||
         (stristr($_SERVER['HTTP_USER_AGENT'], 'NextGenSearchBot')              !== FALSE) ||
         (stristr($_SERVER['HTTP_USER_AGENT'], 'OutfoxBot')                     !== FALSE) ||
         (stristr($_SERVER['HTTP_USER_AGENT'], 'RufusBot')                      !== FALSE) ||
         (stristr($_SERVER['HTTP_USER_AGENT'], 'sogou spider')                  !== FALSE) ||
         (stristr($_SERVER['HTTP_USER_AGENT'], 'Webaroo')                       !== FALSE) ||
         (stristr($_SERVER['HTTP_USER_AGENT'], 'Whirlpool Web Engine')          !== FALSE) ||
         (stristr($_SERVER['HTTP_USER_AGENT'], 'WhoIsBase')                     !== FALSE) ||
        // Generic list...
         (stristr($_SERVER['HTTP_USER_AGENT'], 'bot')                           !== FALSE) ||
         (stristr($_SERVER['HTTP_USER_AGENT'], 'crawl')                         !== FALSE) ||
         (stristr($_SERVER['HTTP_USER_AGENT'], 'spider')                        !== FALSE))) {
        // It is a spider...
        return TRUE;
    } else {
        // Is not a spider...
        return FALSE;
    }
}


/**
* Va-Functions.inc.php :: FixHeartColors()
*
* This routine will take $string and replace any heart codes within the string
* with html color codes.
*
* @param string $string
* @return string
*/
function FixHeartColors($string) {
    if (!is_string($string)) {
        $string = (string) $string;
    }

    // Check to see if the variable was passed, if not then return the string.
    if (trim($string) == '') {
        return $string;
    }

    $spanFlag = 0;                          // Temporary flag for span tagss.

    // Search for hearts.
    for ($position = 0; is_int($position);) {
        $position = strpos($string, chr(3), 0);
        if (is_int($position)) {            // If found a heart...
            $code = substr($string, $position + 1, 1);
            switch (strtoupper($code)) {
                case '0':
                    $color = '#ADAAAD';
                    break;
                case '1':
                    $color = '#52FFFF';
                    break;
                case '2':
                    $color = '#FFFF52';
                    break;
                case '3':
                    $color = '#AD00AD';
                    break;
                case '4':
                    $color = '#FFFFFF';
                    break;
                case '5':
                    $color = '#52FF52';
                    break;
                case '6':
                    $color = '#FF5552';
                    break;
                case '7':
                    $color = '#5255FF';
                    break;
                case '8':
                    $color = '#AD5500';
                    break;
                case '9':
                    $color = '#00AAAD';
                    break;
                case 'A':
                    $color = '#00AA00';
                    break;
                case 'B':
                    $color = '#FF55FF';
                    break;
                case 'C':
                    $color = '#FFFFFF';
                    break;
                case 'D':
                    $color = '#FFFF52';
                    break;
                case 'E':
                    $color = '#52FFFF';
                    break;
                case 'F':
                    $color = '#ADAAAD';
                    break;
                case 'G':
                    $color = '#5255FF';
                    break;
                case 'H':
                    $color = '#FF5552';
                    break;
                case 'I':
                    $color = '#5255FF';
                    break;
                case 'J':
                    $color = '#FFFF52';
                    break;
                case 'K':
                    $color = '#52FFFF';
                    break;
                case 'L':
                    $color = '#ADAAAD';
                    break;
                case 'M':
                    $color = '#FFFFFF';
                    break;
                case 'N':
                    $color = '#ADAAAD';
                    break;
                case 'O':
                    $color = '#ADAAAD';
                    break;
                case 'P':
                    $color = '#AD0000';
                    break;
                case 'Q':
                    $color = '#0000AD';
                    break;
                case 'R':
                    $color = '#FFFFFF';
                    break;
                case 'S':
                    $color = '#FF5552';
                    break;
                case 'T':
                    $color = '#5255FF';
                    break;
                case 'U':
                    $color = '#FFFFFF';
                    break;
                case 'V':
                    $color = '#ADAAAD';
                    break;
                case 'W':
                    $color = '#ADAAAD';
                    break;
                case 'X':
                    $color = '#ADAAAD';
                    break;
                case 'Y':
                    $color = '#FFFFFF';
                    break;
                case 'Z':
                    $color = '#FFFFFF';
                    break;
                case '[':
                    $color = '#525552';
                    break;
                default:
                    $color = '';
                    break;
            }

            if ($color != '') {
                // Insert the new information into the string.
                $a = substr($string, 0, $position);
                $b = substr($string, $position + 2);
                $string = $a . '<span style="color: ' . $color . '">' . $b;

                // Change span flag.
                $spanFlag = 1;

                // Check for next heart code for closing tag.
                $position = strpos($string, chr(3), 0);
                if (is_int($position)) {
                    $a = substr($string, 0, $position);
                    $b = substr($string, $position);
                    $string = $a . '</span>' . $b;
                }
            } else {
                $a = substr($string, 0, $position);
                $b = substr($string, $position + 1);
                $string = $a . $b;
                
                $spanFlag = 0;
            }
        }
    }

    if ($spanFlag == 1) {
        $string = $string . '</span>';
    }

    return $string;                         // Return the new string.
}


/**
* Va-Functions.inc.php :: GetAccFlags()
*
* This function will take the long integer value of the access flags and
* convert it into a string format. The returned letters are all uppercase.
*
* @param integer $accFlagsInt
* @return string
*/
function GetAccFlags($accFlagsInt) {
    if (!is_int($accFlagsInt)) {
        $accFlagsInt = (integer) $accFlagsInt;
    }

    // Check to see if the variable was passed, if not then return nothing.
    if ($accFlagsInt == 0) {
        return '';
    }
    
    $flags  = '';                           // Flags start off as none.
    $num    = 33554432;                     // This is letter Z's value as a user flag.

    for ($letter = 90; $num >= 1; $letter--) {  // It uses capital letters (letter 90).
        $tmp = intval($accFlagsInt / $num);
        if ($tmp >= 1) {
            $flags       = $flags . chr($letter);
            $accFlagsInt = $accFlagsInt - $num;
        }
        $num = $num / 2;
    }

    return strrev($flags);                  // Return the access flags.
}


/**
* Va-Functions.inc.php :: GetChar()
*
* This function will take the given file string and extract the character value
* by using its position.
*
* @param string $fileString
* @param integer $position
* @return integer
*/
function GetChar($fileString, $position) {
    if (!is_string($fileString)) {
        $fileString = (string)  $fileString;
    }
    if (!is_int($position)) {
        $position   = (integer) $position;
    }

    // Check to see if the variable was passed, if not then return nothing.
    if ($fileString == '') {
        return 0;
    }

    // Check to see if the position is larger than the string length.
    if ($position > strlen($fileString)) {
        return 0;
    }

    $position--;                            // Reduce the position by one.
    // Get the first ascii value.
    $a = ord(substr($fileString, $position, 1));

    return (integer) $a;                    // Return the integer.
}


/**
* Va-Functions.inc.php :: GetCurrentURL()
*
* This function simply returns the full URL of the current script. If $query is
* set to 0, then it will return the current URL without the query string.
*
* @param boolean $query
* @return string
*/
function GetCurrentURL($query = 1) {
    if (!is_int($query)) {
        $query = (boolean) $query;
    }

    if (IsHttps()) {
        $scheme = 'https://';
    } else {
        $scheme = 'http://';
    }
    $url = $scheme . $_SERVER['HTTP_HOST'] . $_SERVER['SCRIPT_NAME'];
    if (strpos($url, 'php.exe') !== FALSE) {
        $url = $scheme . $_SERVER['HTTP_HOST'] . $_SERVER['PHP_SELF'];
    }
    if ((isset($_SERVER['QUERY_STRING'])) &&
        ($query) &&
        (trim($_SERVER['QUERY_STRING']) != '')) {
        $url = $url . '?' . $_SERVER['QUERY_STRING'];
    }

    return $url;                            // Return the url.
}


/**
* Va-Functions.inc.php :: GetInteger()
*
* This function will take the given file string and extract the integer value
* by using its position. If you set $unsigned to 1, then you can extract
* values up to 65535 that are positive only.
*
* @param string $fileString
* @param integer $position
* @param boolean $unsigned
* @return integer
*/
function GetInteger($fileString, $position, $unsigned = 0) {
    if (!is_string($fileString)) {
        $fileString = (string)  $fileString;
    }
    if (!is_int($position)) {
        $position   = (integer) $position;
    }
    if (!is_bool($unsigned)) {
        $unsigned   = (boolean) $unsigned;
    }

    // Check to see if the variable was passed, if not then return nothing.
    if ($fileString == '') {
        return 0;
    }

    // Check to see if the position is larger than the string length.
    if ($position > strlen($fileString)) {
        return 0;
    }

    $position--;                            // Reduce the position by one.
    // Get the first ascii value.
    $a = ord(substr($fileString, $position, 1));

    $position++;                            // Move to the next position.
    // Get the second ascii value.
    $b = 256 * ord(substr($fileString, $position, 1));
    
    $integer = intval($a + $b);             // Create the integer.
    
    // Check for negative value.
    if ((!$unsigned) && ($integer > 32767)) {
        $integer = $integer - 65535 - 1;
    }

    return $integer;                        // Return the integer.
}


/**
* Va-Functions.inc.php :: GetLongInt()
*
* This function will take the given file string and extract the long integer
* value by using its position. If you set $unsigned to 1, then you can extract
* values up to 4294967295 that are positive only.
*
* @param string $fileString
* @param integer $position
* @param boolean $unsigned
* @return integer
*/
function GetLongInt($fileString, $position, $unsigned = 0) {
    if (!is_string($fileString)) {
        $fileString = (string)  $fileString;
    }
    if (!is_int($position)) {
        $position   = (integer) $position;
    }
    if (!is_bool($unsigned)) {
        $unsigned   = (boolean) $unsigned;
    }

    // Check to see if the variable was passed, if not then return nothing.
    if ($fileString == '') {
        return 0;
    }
    
    // Check to see if the position is larger than the string length.
    if ($position > strlen($fileString)) {
        return 0;
    }
    
    $position--;                            // Reduce the position by one.
    // Get the first ascii value.
    $a = ord(substr($fileString, $position, 1));
    
    $position++;                            // Move to the next position.
    // Get the second ascii value.
    $b = 256 * ord(substr($fileString, $position, 1));

    $position++;                            // Move to the next position.
    // Get the third ascii value.
    $c = 65536 * ord(substr($fileString, $position, 1));
    
    $position++;                            // Move to the next position.
    // Get the fourth ascii value.
    $d = 16777216 * ord(substr($fileString, $position, 1));
    
    $integer = intval($a + $b + $c + $d);   // Create the integer.
    
    // Check for negative value.
    if ((!$unsigned) && ($integer > 2147483647)) {
        $integer = $integer - 2147483647 - 1;
    }

    return $integer;                        // Return the long integer.
}


/**
* Va-Functions.inc.php :: GetSingle()
*
* This function will take the given file string and extract the single precison
* value by using its position. If $round is set to any postitive value, then
* the answer will be rounded off to that many decimal places.
*
* This function does not work 100% correctly, especially for larger values. It
* correctly extracts the VADV values, but does not reliably calculate other
* values that may equal or exceed thousands.
*
* Information to do these calculations were taken from:
* http://www.learnbinary.com/binary.html
* http://www.aplawrence.com/Basics/floatingpoint.html
*
* @param string $fileString
* @param integer $position
* @param integer $round
* @return float
*/
function GetSingle($fileString, $position, $round = -1) {
    if (!is_string($fileString)) {
        $fileString = (string)  $fileString;
    }
    if (!is_int($position)) {
        $position   = (integer) $position;
    }
    if (!is_int($round)) {
        $round      = (integer) $round;
    }

    // Save the current precision setting...
    $precision = ini_get('precision');
    ini_set('precision', 40);

    // Extract the four bytes, convert to binary and pad them with zeros if needed.
    $a = str_pad(decbin(GetChar($fileString, $position)), 8, 0, STR_PAD_LEFT);
    $b = str_pad(decbin(GetChar($fileString, $position + 1)), 8, 0, STR_PAD_LEFT);
    $c = str_pad(decbin(GetChar($fileString, $position + 2)), 8, 0, STR_PAD_LEFT);
    $d = str_pad(decbin(GetChar($fileString, $position + 3)), 8, 0, STR_PAD_LEFT);

    // Put together the 32 bit binary number.
    $long = $d . $c . $b . $a;

    // Break into separate parts.
    $sign = substr($long, 0, 1);
    $exp  = bindec(substr($long, 1, 8)) - 127;
    $mant = 1 . '.' . substr($long, 9);

    // Calculate the binary answer.
    if ($exp >= 0) {                        // If positive, do the easy way.
        $float = $mant * decbin(pow(2, $exp));
    } else {
        $float = $mant * (10 / (str_pad(1, 2 - $exp, 0, STR_PAD_RIGHT)));
    }

    // If the negative exp is too long, PHP does it in abbreviated notation,
    // so I'm converting it to a string in order to do the rest of the
    // calculations.
    $float = strtoupper($float);
    $pos   = strpos($float, 'E');
    if ($pos !== FALSE) {
        $pol = substr($float, $pos, 1);
        $pow = (integer) substr($float, $pos + 1);
        if ($pow < 0) {
            $pow = substr($pow, 1);
        }
        $float = substr($float, 0, $pos);
        $float = str_replace('.', '', $float);
        if ($pol == '-') {
            $float = '0.' . str_pad($float, $pow - 1 + strlen($float), '0', STR_PAD_LEFT);
        } else {
            if ($pow > strlen($float)) {
                $float = str_pad($float, $pow + 1, '0', STR_PAD_RIGHT);
            } else {
                $float = substr($float, 0, $pow) . '.' . substr($float, $pow + 1);
            }
        }
    }
    
    // Split the number into an integer and decimal.
    $array   = explode('.', $float);
    $integer = bindec($array[0]);
    if (!isset($array[1])) {
        $array[1] = '';
    }
    
    // Calculate the fractions of the decimal part (which is in binary).
    // This is the longer, harder way...
    $decimal  = 0;
    $digit    = 1;
    $fraction = 2;
    for ($x = 0; $digit != ''; $x++) {
        $digit = substr($array[1], $x, 1);
        if ($digit != '') {
            $decimal = $decimal + ((1 / $fraction) * $digit);
        }
        $fraction = $fraction * 2;
    }

    // This is the shorter, more compact way...
    // $decimal = bindec($array[1]) / pow(2, strlen($array[1]));

    // Put the two together to get your answer.
    $float = $integer + $decimal;

    // Round the answer if specified.
    if ($round >= 0) {
        $float = round($float, $round);
    } else {
        $float = round($float, 10);
    }

    // Add on the sign if negative.
    if ($sign == 1) {
        $float = 0 - $float;
    }

    // Set the precision back.
    ini_set('precision', $precision);

    return (float) $float;                  // Return the floating point number.
}


/**
* Va-Functions.inc.php :: GetString()
*
* This function will take the given file string and extract the string value
* by using its position and length.
*
* @param string $fileString
* @param integer $position
* @param integer $length
* @return string
*/
function GetString($fileString, $position, $length) {
    if (!is_string($fileString)) {
        $fileString = (string)  $fileString;
    }
    if (!is_int($position)) {
        $position   = (integer) $position;
    }
    if (!is_int($length)) {
        $length     = (integer) $length;
    }

    // Check to see if the variable was passed, if not then return nothing.
    if ($fileString == '') {
        return '';
    }

    // Check to see if the position is larger than the string length.
    if ($position > strlen($fileString)) {
        return '';
    }

    $position--;                            // Reduce the position by one.
    $string = trim(substr($fileString, $position, $length));

    return (string) $string;                // Return the string.
}


/**
* Va-Functions.inc.php :: GlobalFooter()
*
* A footer for all pages. Displays execution time and copyright information.
* For dark backgrounds, set $color to 1.
*
* @param boolean $color
* @param boolean $noBar
*/
function GlobalFooter($color = 0, $noBar = 0) {
    if (!is_bool($color)) {
        $color = (boolean) $color;
    }
    if (!is_bool($noBar)) {
        $noBar = (boolean) $noBar;
    }

    global $CONFIG;
    global $VERSION;
    global $startTime;

    $totalTime = round(ProgramTime() - $startTime, 4);
    if (!$color) {
?>

<style type="text/css">
<!--
a.footerBoldBody:link    { font-family: Verdana, Arial, Helvetica; font-size: 8pt;  color: #000000; font-weight:bold }
a.footerBoldBody:visited { font-family: Verdana, Arial, Helvetica; font-size: 8pt;  color: #000000; font-weight:bold }
.footerBody              { font-family: Verdana, Arial, Helvetica; font-size: 8pt;  color: #000000 }
.footerBoldBody          { font-family: Verdana, Arial, Helvetica; font-size: 8pt;  color: #000000; font-weight:bold }
.footerVersion           { font-family: Courier New;               font-size: 8pt;  color: #808080; font-weight:bold }
-->
</style>

<?php
if (!$noBar) {
    print '<hr color="#000080">';
} else {
    print '<br />';
}
?>
<div align="center">
    <center>
    <table bgcolor="#FFFFFF" border="0" cellpadding="5" cellspacing="0" style="border-collapse: collapse" bordercolor="#111111" width="100%">
        <tr>
            <td width="100%" align="center" valign="top">
                <p><span class="footerBoldBody">This page is powered by <a class="footerBoldBody" href="http://www.vadvphp.com">VADV-PHP</a></span><br /><span class="footerBody">Execution Time: <?php print $totalTime; ?> seconds</span></p>
                <p class="footerBody">If you experience any problems with this website or need help, contact the <a href="mailto:<?php print $CONFIG['webmaster']; ?>">webmaster</a>.<br /> VADV-PHP Copyright  2002-<?php print date('Y'); ?> Steve Winn. All Rights Reserved.<br /> Virtual Advanced Copyright  1995-1997 Roland De Graaf.<br /><span class="footerVersion">v<?php print $VERSION; ?></span></p>
            </td>
        </tr>
    </table>
    </center>
</div>

<?php
    } else {
?>

<style type="text/css">
<!--
a.footerBoldBody:link    { font-family: Verdana, Arial, Helvetica; font-size: 8pt;  color: #FFFFFF; font-weight:bold }
a.footerBoldBody:visited { font-family: Verdana, Arial, Helvetica; font-size: 8pt;  color: #FFFFFF; font-weight:bold }
.footerBody              { font-family: Verdana, Arial, Helvetica; font-size: 8pt;  color: #FFFFFF }
.footerBoldBody          { font-family: Verdana, Arial, Helvetica; font-size: 8pt;  color: #FFFFFF; font-weight:bold }
.footerVersion           { font-family: Courier New;               font-size: 8pt;  color: #C0C0C0; font-weight:bold }
-->
</style>

<hr color="#525552">
<div align="center">
    <center>
    <table bgcolor="#000000" border="0" cellpadding="5" cellspacing="0" style="border-collapse: collapse" bordercolor="#111111" width="100%">
        <tr>
            <td width="100%" align="center" valign="top">
                <p><span class="footerBoldBody">This page is powered by <a class="footerBoldBody" href="http://www.vadvphp.com">VADV-PHP</a></span><br /><span class="footerBody">Execution Time: <?php print $totalTime; ?> seconds</span></p>
                <p class="footerBody">If you experience any problems with this website or need help, contact the <a href="mailto:<?php print $CONFIG['webmaster']; ?>">webmaster</a>.<br /> VADV-PHP Copyright  2002-<?php print date('Y'); ?> Steve Winn. All Rights Reserved.<br /> Virtual Advanced Copyright  1995-1997 Roland De Graaf.<br /><span class="footerVersion">v<?php print $VERSION; ?></span></p>
            </td>
        </tr>
    </table>
    </center>
</div>

<?php
    }
}


/**
* Va-Functions.inc.php :: hmac_md5()
*
* Made by Jakub Vrna, jakub@vrana.cz
* Need JavaScript MD5 library made by Paul Johnston, http://pajhome.org.uk/crypt/md5/
*
* @param string $key
* @param string $data
* @return string
*/
function hmac_md5($key, $data) {
	if (extension_loaded('mhash')) {
		return bin2hex(mhash(MHASH_MD5, $data, $key));
	}

	// RFC 2104 HMAC implementation for php. Hacked by Lance Rushing
	$b = 64;                                   // byte length for md5
	if (strlen($key) > $b) {
		$key = pack('H*', md5($key));
	}
	$key = str_pad($key, $b, chr(0x00));
	$ipad = str_pad('', $b, chr(0x36));
	$opad = str_pad('', $b, chr(0x5c));
	$k_ipad = $key ^ $ipad;
	$k_opad = $key ^ $opad;

	return md5($k_opad . pack('H*', md5($k_ipad . $data)));
}


/**
* Va-Functions.inc.php :: hmac_sha1()
*
* Made by Jakub Vrna, jakub@vrana.cz
* Need JavaScript SHA1 library made by Paul Johnston, http://pajhome.org.uk/crypt/md5/
*
* @param string $key
* @param string $data
* @return string
*/
function hmac_sha1($key, $data) {
	// RFC 2104 HMAC implementation for php. Hacked by Lance Rushing
	$b = 64;                                   // byte length for md5
	if (strlen($key) > $b) {
		$key = pack('H*', sha1($key));
	}
	$key = str_pad($key, $b, chr(0x00));
	$ipad = str_pad('', $b, chr(0x36));
	$opad = str_pad('', $b, chr(0x5c));
	$k_ipad = $key ^ $ipad;
	$k_opad = $key ^ $opad;

	return sha1($k_opad . pack('H*', sha1($k_ipad . $data)));
}


/**
* Va-Functions.inc.php :: HtmlEncode()
*
* This function will take the passed line and add html href information for any
* links found. In basic mode, it will search for mailto, ftp, http and telnet.
* In enhanced mode, it will search for unregistered urls.
*
* @param string $string
* @return string
*/
function HTMLEncode($string, $enhanced = 0) {
    // http://www.w3.org/Addressing/schemes
    if (!is_string($string)) {
        $string = (string) $string;
    }
    if (!is_bool($enhanced)) {
        $enhanced = (boolean) $enhanced;
    }

    $string = ' ' . $string;

    if (!$enhanced) {
        $string = preg_replace(array(
                                     "/([\s|\r\n|\r|\n|\(|\[\{\-:;\'\"\<.,?])(([\w\.\-]+))(@)([\w\.\-]+)\b\.([\w\.\-]+)\b/i",
                                     "/([\s|\r\n|\r|\n|\(|\[\{\-:;\'\"\<.,?])([f|F][t|T][p|P]:\/\/([\w~:@]+))([\w\.\/\&\=\?\-~:;%]+)/",
                                     "/([\s|\r\n|\r|\n|\(|\[\{\-:;\'\"\<.,?])([f|F][t|T][p|P]\.)([\w\.\/\&\=\?\-~:;%]+)/",
                                     "/([\s|\r\n|\r|\n|\(|\[\{\-:;\'\"\<.,?])([h|H][t|T][t|T][p|P](s?):\/\/)([\w\.\/\&\=\?\-\+\(\),~:#%]+)/",
                                     "/([\s|\r\n|\r|\n|\(|\[\{\-:;\'\"\<.,?])([w|W][w|W][w|W]\.)([\w\.\/\&\=\?\-\+\(\),~:#%]+)/",
                                     "/([\s|\r\n|\r|\n|\(|\[\{\-:;\'\"\<.,?])([t|T][e|E][l|L][n|N][e|E][t|T]:\/\/([\w~:@]+))([\w\.\/\&\=\?\-:]+)/"
                                    ),
                               array(
                                     "$1<a href=\"mailto:$3$4$5.$6\">$3$4$5.$6</a>",
                                     "$1<a href=\"ftp://$3$4\" target=\"_blank\">$2$4</a>",
                                     "$1<a href=\"ftp://$2$3$4\" target=\"_blank\">$2$3$4</a>",
                                     "$1<a href=\"http$3://$4$5\" target=\"_blank\">$2$4$5</a>",
                                     "$1<a href=\"http://$2$3\" target=\"_blank\">$2$3</a>",
                                     "$1<a href=\"telnet://$3$4\">$2$4</a>"
                                    ),
                                    $string);
    } else {
        $string = preg_replace(array(
                                     "/([\s|\r\n|\r|\n|\(|\[\{\-:;\'\"\<.,?])(([\w\.\-]+))(@)([\w\.\-]+)\b\.([\w\.\-]+)\b/i",
                                     "/([\s|\r\n|\r|\n|\(|\[\{\-:;\'\"\<.,?])([f|F][t|T][p|P]:\/\/([\w~:@]+))([\w\.\/\&\=\?\-~:;%]+)/",
                                     "/([\s|\r\n|\r|\n|\(|\[\{\-:;\'\"\<.,?])([f|F][t|T][p|P]\.)([\w\.\/\&\=\?\-~:;%]+)/",
                                     "/([\s|\r\n|\r|\n|\(|\[\{\-:;\'\"\<.,?])([h|H][t|T][t|T][p|P](s?):\/\/)([\w\.\/\&\=\?\-\+\(\),~:#%]+)/",
                                     "/([\s|\r\n|\r|\n|\(|\[\{\-:;\'\"\<.,?])([w|W][w|W][w|W]\.)([\w\.\/\&\=\?\-\+\(\),~:#%]+)/",
                                     "/([\s|\r\n|\r|\n|\(|\[\{\-:;\'\"\<.,?])([t|T][e|E][l|L][n|N][e|E][t|T]:\/\/([\w~:@]+))([\w\.\/\&\=\?\-:]+)/",
                                     "/([\s|\r\n|\r|\n|\(|\[\{\-:;\'\"\<.,?])([g|G][o|O][p|P][h|H][e|E][r|R]:\/\/)([\w\.\/\&\=\?\-:|%]+)/",
                                     "/([\s|\r\n|\r|\n|\(|\[\{\-:;\'\"\<.,?])([n|N][e|E][w|W][s|S]:)([\w\.\/\&\=\?\-:*]+)/",
                                     "/([\s|\r\n|\r|\n|\(|\[\{\-:;\'\"\<.,?])([n|N][n|N][t|T][p|P]:\/\/)([\w.\/\&\=\?\-~:#]+)/",
                                     "/([\s|\r\n|\r|\n|\(|\[\{\-:;\'\"\<.,?])([a|A][i|I][m|M]:)([\w\.\/\&\=\?\-:#]+)/"
                                    ),
                               array(
                                     "$1<a href=\"mailto:$3$4$5.$6\">$3$4$5.$6</a>",
                                     "$1<a href=\"ftp://$3$4\" target=\"_blank\">$2$4</a>",
                                     "$1<a href=\"ftp://$2$3$4\" target=\"_blank\">$2$3$4</a>",
                                     "$1<a href=\"http$3://$4$5\" target=\"_blank\">$2$4$5</a>",
                                     "$1<a href=\"http://$2$3\" target=\"_blank\">$2$3</a>",
                                     "$1<a href=\"telnet://$3$4\">$2$4</a>",
                                     "$1<a href=\"gopher://$3\">$2$3</a>",
                                     "$1<a href=\"news:$3\">$2$3</a>",
                                     "$1<a href=\"nntp://$3\">$2$3</a>",
                                     "$1<a href=\"aim:$3\">$2$3</a>"
                                    ),
                                    $string);
    }

    return substr($string, 1);              // Return the encoded string, remove that inital space.
}


/**
* Va-Functions.inc.php :: HtmlSpace()
*
* Add non-breaking spaces to a string for output.
*
* @param string $string
* @return string
*/
function HtmlSpace($string) {
    if (!is_string($string)) {
        $string = (string) $string;
    }

    $string = str_replace('  ', '&nbsp;&nbsp;', $string);
    return $string;
}


/**
* Va-Functions.inc.php :: IsHttps()
*
* Determines if the current page's protocol.
*
* @return boolean
*/
function IsHttps() {
    if ((isset($_SERVER['HTTPS'])) &&
        (strtolower(trim($_SERVER['HTTPS'])) == 'on')) {
        return TRUE;
    } else {
        return FALSE;
    }
}


/**
* Va-Functions.inc.php :: MakeAccFlags()
*
* This function will take the string representation of user access flags and
* convert it into an integer suitable for writing to userfile.dat
*
* @param string $accFlags
* @return integer
*/
function MakeAccFlags($accFlags) {
    if (!is_string($accFlags)) {
        $accFlags = (string) $accFlags;
    }

    $accFlags    = strtoupper(str_replace(' ', '', $accFlags)); // Remove any white space and convert it to upper case.
    $accFlagsInt = 0;
    
    for ($x = strlen($accFlags); $x >= 0; $x--) {
        $letter = substr($accFlags, $x, 1); // Get a character.
        if (((ord($letter) >= 65) &&                            // It must be a letter, or ignore it.
             (ord($letter) <= 90)) &&
            (strpos($accFlags, $letter) === $x)) {              // If the letter does not occur before this one...
            $accFlagsInt = $accFlagsInt + pow(2, ord($letter) - 65);
        }
    }
    
    return $accFlagsInt;                    // Return the integer.
}


/**
* Va-Functions.inc.php :: MakeChar()
*
* This function will take the value given and convert it into a one byte
* integer represented by ascii characters.
*
* @param integer $value
* @return string
*/
function MakeChar($value) {
    if (!is_int($value)) {
        $value = (integer) $value;
    }

    if ($value == 0) {
        return chr(0);
    }
    if ($value > 256) {
        trigger_error(VA_Error(2));
    }
    
    $a = chr($value);                       // Set the variable to the ascii character.
    
    return $a;                              // Return the character.
}


/**
* Va-Functions.inc.php :: MakeFlags()
*
* This function will take the string $accFlags and convert it to the 26 character
* representation which is writable to userfile.dat.
*
* @param string $accFlags
* @return string
*/
function MakeFlags($accFlags) {
    if (!is_string($accFlags)) {
        $accFlags = (string) $accFlags;
    }

    $accFlags = strtoupper(str_replace(' ', '', $accFlags));    // Remove any white space and convert it to upper case.
    $newFlags = str_pad('', 26, ' ');                           // Create the blank string.

    // Check to see if the variable was passed, if not then return nothing.
    if ($accFlags == '') {
        return $newFlags;
    }
    
    for ($x = strlen($accFlags); $x >= 0; $x--) {
        $letter = substr($accFlags, $x, 1); // Get a character.
        if ((ord($letter) >= 65) && (ord($letter) <= 90)) {     // It must be a letter, or ignore it.
            $newFlags = substr_replace($newFlags, $letter, ord($letter) - 65, 1);
        }
    }
    
    return $newFlags;                       // Return the new string.
}


/**
* Va-Functions.inc.php :: MakeInteger()
*
* This function will take the value given and convert it into a two byte
* integer represented by ascii characters. The values can be from -32768 to
* 32767. If you set $unsigned to 1, then you can create an integer with a value
* up to 65535.
*
* @param integer $value
* @param boolean $unsigned
* @return string
*/
function MakeInteger($value, $unsigned = 0) {
    if (!is_int($value)) {
        $value    = (integer) $value;
    }
    if (!is_bool($unsigned)) {
        $unsigned = (boolean) $unsigned;
    }

    if ($value == 0) {
        return chr(0) . chr(0);
    }
    if (!$unsigned) {
        if ($value > 32767) {               // This is the variable type maximum.
            trigger_error(VA_Error(2));
        }
        if ($value < -32768) {
            trigger_error(VA_Error(2));
        }
        if ($value < 0) {
            $value = 65535 + $value + 1;
        }
    } else {
        if ($value > 65535) {               // This is the maximum that can be stored.
            trigger_error(VA_Error(2));
        }
        if ($value < 0) {
            trigger_error(VA_Error(2));
        }
    }
    
    $b = intval($value / 256);              // Divide the value.
    $value = $value - ($b * 256);           // Subtract the variable from the value.
    $b = chr($b);                           // Set the variable to the ascii character.

    $a = chr($value);                       // Set the variable to the ascii character.

    return $a . $b;                         // Return the ascii representation.
}


/**
* Va-Functions.inc.php :: MakeLongInt()
*
* This function will take the value given and convert it into a four byte
* long integer represented by ASCII characters. The values can be from
* -2147483648 to 2147483647. If you set $unsigned to 1, then you can create a
* long integer with a value up to 4294967295.
*
* @param integer $value
* @param boolean $unsigned
* @return string
*/
function MakeLongInt($value, $unsigned = 0) {
    if (!is_int($value)) {
        $value    = (integer) $value;
    }
    if (!is_bool($unsigned)) {
        $unsigned = (boolean) $unsigned;
    }

    if ($value == 0) {
        return chr(0) . chr(0) . chr(0) . chr(0);
    }
    if (!$unsigned) {
        if ($value > 2147483647) {          // This is the variable type maximum.
            trigger_error(VA_Error(2));
        }
        if ($value < -2147483648) {
            trigger_error(VA_Error(2));
        }
        if ($value < 0) {
            $value = 2147483647 + $value + 1;
        }
    } else {
        if ($value > 4294967295) {          // This is the maximum that can be stored.
            trigger_error(VA_Error(2));
        }
        if ($value < 0) {
            trigger_error(VA_Error(2));
        }
    }
    
    $d = intval($value / 16777216);         // Divide the value.
    $value = $value - ($d * 16777216);      // Subtract the variable from the value.
    $d = chr($d);                           // Set the variable to the ascii character.

    $c = intval($value / 65536);            // Divide the value.
    $value = $value - ($c * 65536);         // Subtract the variable from the value.
    $c = chr($c);                           // Set the variable to the ascii character.
    
    $b = intval($value / 256);              // Divide the value.
    $value = $value - ($b * 256);           // Subtract the variable from the value.
    $b = chr($b);                           // Set the variable to the ascii character.
    
    $a = chr($value);                       // Set the variable to the ascii character.

    return $a . $b . $c . $d;               // Return the ascii representation.
}


/**
* Va-Functions.inc.php :: MakeSingle()
*
* This function will take the floating point $value and convert it into a four
* byte ASCII representation. The maximum is 3.40282e+38.
*
* @param float $value
* @return string
*/
function MakeSingle($value) {
    if (!is_float($value)) {
        $value = (float) $value;
    }

    // Determine the sign.
    if ($value < 0) {
        $sign = 1;
    } else {
        $sign = 0;
    }

    // Split the number up into parts.
    $array = explode('.', $value);
    $integer = decbin($array[0]);
    $decimal = '';

    // Create the binary representation of the decimal.
    if (isset($array[1])) {
        $dec = '.' . $array[1];
        $fraction = 0;
        for ($x = 0; (($x < 10) && ($fraction != 1)) ; $x++) {
            $fraction = $dec * 2;
            if ($fraction >= 1) {
                $decimal = $decimal . '1';
                $dec = $fraction - 1;
            } else {
                $decimal = $decimal . '0';
                $dec = $fraction;
            }
        }
    }

    if ($integer > 0) {
        $exp = strlen($integer) - 1;
        $integer = substr($integer, 1);
        $float = $integer . $decimal;
    } else {
        if ($decimal != 0) {
            $exp = strpos($decimal, '1') + 1;
            $float = substr($decimal, $exp);
            $exp = '-' . $exp;
        } else {
            $exp = -127;
            $float = 0;
        }
    }

    $exp = decbin($exp + 127);
    $exp = str_pad($exp, 8, '0', STR_PAD_LEFT);

    $float = str_pad($float, 23, '0', STR_PAD_RIGHT);
    $float = $sign . $exp . $float;

    // Create the four bytes.
    $d = bindec(substr($float, 0, 8));
    $d = chr($d);
    $c = bindec(substr($float, 8, 8));
    $c = chr($c);
    $b = bindec(substr($float, 16, 8));
    $b = chr($b);
    $a = bindec(substr($float, 24, 8));
    $a = chr($a);

    return $a . $b . $c . $d;               // Return the ascii representation.
}


/**
* Va-Functions.inc.php :: ParseANSI()
*
* This function will parse an ANSI coded file and return it as a HTML string.
* It uses the Windows Termainal font, so it may not display correctly on non-
* Windows machines.
*
* You can specify the font-size the final result will be encoded to use. You can
* set the maximum number of rows to display (up to 1000). If 'truncate' is
* turned on, then any blank rows at the end of the file will be removed.
*
* @param string $file
* @return string
*/
function ParseANSI($fileString, $fontSize = 12, $maxRows = 25, $truncate = 1) {
    if (!is_string($fileString)) {
        $fileString = (string) $fileString;
    }
    if (!is_int($fontSize)) {
        $fontSize = (integer) $fontSize;
    }
    if (!is_int($maxRows)) {
        $maxRows = (integer) $maxRows;
    }
    if (!is_bool($truncate)) {
        $truncate = (boolean) $truncate;
    }

    // Check the maxrows.
    if ($maxRows < 1) {
        $maxRows = 1;
    }
    if ($maxRows > 1000) {
        $maxRows = 1000;
    }
    $maxSize = $maxRows * 80;

    // Remove any codes that will not be processed.
    $pattern = array(
                     "/\e\[((\?*?(\d){1,2})|((\d){1,2};{0,1})){0,2}[R|h|l|y]/",
                     "/\e\[(((\d){1,2};{0,1}){0,4})[p]/",
                     "/\e\[6[n]/",
                     "/\e\[((\d){1,2};{0,1}){0,3}[@|L|M|P]/",
                    );
    $fileString = preg_replace($pattern, '', $fileString);

    // Two arrays are created. One being the style array, which contains the
    // information about attributes to the block. The other array contains the
    // content to display in the block.
    // 80 x 25 = 2000 blocks

    // Style array contains attribute information. It is in this format:
    // Bold ; Dim ; Underline ; Blink ; Reverse ; Invisible ; Foreground ; Background
    // By default, all blocks are set to 0;0;0;0;0;0;37;40
    $style = array();
    for ($x = 0; $x < $maxSize; $x++) {
        $style[$x] = '0;0;0;0;0;0;37;40';
    }

    // Content array contains ASCII characters. By default, all blocks contain
    // a space.
    $content = array();
    for ($x = 0; $x < $maxSize; $x++) {
        $content[$x] = ' ';
    }

    $escapeCode = chr(27) . '[';
    $currentStyle = '0;0;0;0;0;0;37;40';

    $pos = 0;                               // X position.
    $arrayPos = 0;                          // Position in the array.
    $savedPos = '0;0';                      // Saved position.
    for ($x = 0; strlen($fileString) > 0; $x++) {
        $match = strpos($fileString, $escapeCode);
        if ($arrayPos > $maxSize - 1) {
            $arrayPos = $maxSize - 1;
        }
        if ($match === FALSE) {
            // If no match, then place the rest of the fileString into the array
            // one character at a time, up to the $maxSize.
            for ($arrayPos; (($arrayPos < $maxSize) && (strlen($fileString) > 0)); $arrayPos++) {
                $char = substr($fileString, 0, 1);
                if ($char != "\r") {
                    if ($char == "\n") {
                        $arrayPos = $arrayPos - $pos + 78;
                        $pos = 0;
                    } else {
                        $style[$arrayPos]   = $currentStyle;
                        $content[$arrayPos] = $char;
                        if ($pos < 80) {
                            $pos++;
                        } else {
                            $pos = 0;
                        }
                    }
                }
                $fileString = substr($fileString, 1);
            }
            $fileString = '';
        } else if ($match > 0) {
            // If the code is not at the beginning, then enter in the chars
            // up to the first code.
            for ($y = 0; $y < $match; $y++) {
                $char = substr($fileString, 0, 1);
                if ($char != "\r") {
                    if ($char == "\n") {
                        $arrayPos = $arrayPos - $pos + 78;
                        $pos = 0;
                    } else {
                        $style[$arrayPos]   = $currentStyle;
                        $content[$arrayPos] = $char;
                        if ($pos < 80) {
                            $pos++;
                        } else {
                            $pos = 0;
                        }
                    }
                }
                $arrayPos++;
                $fileString = substr($fileString, 1);
            }
        } else {
            $foundMatch = 0;
            if ((preg_match("/\e\[(\d){0,3}[A]/", $fileString, $code)) &&
                ($foundMatch == 0)) {
                if (strpos($fileString, $code[0]) === 0) {
                    $foundMatch = 1;
                    $max = intval($arrayPos / 80);
                    $length = strpos($fileString, 'A');
                    $length = (integer) substr($fileString, 2, $length - 2);
                    if ($length < 1) {
                        $length = 1;
                    }
                    if ($max < $length) {
                        $arrayPos = $arrayPos - ($max * 80);
                    } else {
                        $arrayPos = $arrayPos - ($length * 80);
                    }
                    $fileString = preg_replace("/\e\[(\d){0,3}[A]/", '', $fileString, 1);
                }
            }
            if ((preg_match("/\e\[(\d){0,3}[B]/", $fileString, $code)) &&
                ($foundMatch == 0)) {
                if (strpos($fileString, $code[0]) === 0) {
                    $foundMatch = 1;
                    $max = intval(($maxSize - $arrayPos) / 80) - 1;
                    $length = strpos($fileString, 'B');
                    $length = (integer) substr($fileString, 2, $length - 2);
                    if ($length < 1) {
                        $length = 1;
                    }
                    if ($max < $length) {
                        $arrayPos = $arrayPos + ($max * 80);
                    } else {
                        $arrayPos = $arrayPos + ($length * 80);
                    }
                    $fileString = preg_replace("/\e\[(\d){0,3}[B]/", '', $fileString, 1);
                }
            }
            if ((preg_match("/\e\[(\d){0,3}[C]/", $fileString, $code)) &&
                ($foundMatch == 0)) {
                if (strpos($fileString, $code[0]) === 0) {
                    $foundMatch = 1;
                    $max = 79 - $pos;
                    $length = strpos($fileString, 'C');
                    $length = (integer) substr($fileString, 2, $length - 2);
                    if ($length < 1) {
                        $length = 1;
                    }
                    if ($max < $length) {
                        $arrayPos = $arrayPos + $max;
                        $pos = $pos + $max;
                    } else {
                        $arrayPos = $arrayPos + $length;
                        $pos = $pos + $length;
                    }
                    $fileString = preg_replace("/\e\[(\d){0,3}[C]/", '', $fileString, 1);
                }
            }
            if ((preg_match("/\e\[(\d){0,3}[D]/", $fileString, $code)) &&
                ($foundMatch == 0)) {
                if (strpos($fileString, $code[0]) === 0) {
                    $foundMatch = 1;
                    $max = $pos;
                    $length = strpos($fileString, 'D');
                    $length = (integer) substr($fileString, 2, $length - 2);
                    if ($length < 1) {
                        $length = 1;
                    }
                    if (0 > $pos - $length) {
                        $arrayPos = $arrayPos - $max;
                        $pos = $pos - $max;
                    } else {
                        $arrayPos = $arrayPos - $length;
                        $pos = $pos - $length;
                    }
                    $fileString = preg_replace("/\e\[(\d){0,3}[D]/", '', $fileString, 1);
                }
            }
            if ((preg_match("/\e\[[s]/", $fileString, $code)) &&
                ($foundMatch == 0)) {
                if (strpos($fileString, $code[0]) === 0) {
                    $savedPos = $arrayPos . ';' . $pos;
                    $fileString = preg_replace("/\e\[[s]/", '', $fileString, 1);
                }
            }
            if ((preg_match("/\e\[[u]/", $fileString, $code)) &&
                ($foundMatch == 0)) {
                if (strpos($fileString, $code[0]) === 0) {
                    $restorePos = explode(';', $savedPos);
                    $arrayPos = $restorePos[0];
                    $pos = $restorePos[1];
                    $fileString = preg_replace("/\e\[[u]/", '', $fileString, 1);
                }
            }
            if ((preg_match("/\e\[((\?*?(\d){1,2})|((\d){1,2};{0,1})){0,2}[H|f]/", $fileString, $code)) &&
                ($foundMatch == 0)) {
                if (strpos($fileString, $code[0]) === 0) {
                    if (isset($code[5])) {
                        $y = (integer) $code[4];
                        $x = (integer) $code[2];
                    } else if (isset($code[2])) {
                        $y = (integer) $code[2];
                        $x = 0;
                    } else {
                        $y = 0;
                        $x = 0;
                    }

                    $y--;
                    $x--;

                    if ($y < 0) {
                        $y = 0;
                    }
                    if ($x < 0) {
                        $x = 0;
                    }

                    $arrayPos = (80 * $y) + $x;
                    $pos = $x;
                    $fileString = preg_replace("/\e\[((\?*?(\d){1,2})|((\d){1,2};{0,1})){0,2}[H|f]/", '', $fileString, 1);
                }
            }
            if ((preg_match("/\e\[2J/", $fileString, $code)) &&
                ($foundMatch == 0)) {
                if (strpos($fileString, $code[0]) === 0) {
                    // Reset the entire page.
                    for ($x = 0; $x < $maxSize; $x++) {
                        $style[$x] = '0;0;0;0;0;0;37;40';
                    }

                    for ($x = 0; $x < $maxSize; $x++) {
                        $content[$x] = ' ';
                    }

                    $currentStyle = '0;0;0;0;0;0;37;40';
                    $pos = 0;
                    $arrayPos = 0;
                    $fileString = preg_replace("/\e\[2J/", '', $fileString, 1);
                }
            }
            if ((preg_match("/\e\[[K]/", $fileString, $code)) &&
                ($foundMatch == 0)) {
                if (strpos($fileString, $code[0]) === 0) {
                    $length = (79 - $pos) + 1;
                    for ($y = 0; $y < $length; $y++) {
                        $style[$arrayPos + $y] = '0;0;0;0;0;0;37;40';
                        $content[$arrayPos + $y] = ' ';
                    }
                    $fileString = preg_replace("/\e\[[K]/", '', $fileString, 1);
                }
            }
            if ((preg_match("/\e\[(((\d){1,2};{0,1}){0,4})[m]/", $fileString, $code)) &&
                ($foundMatch == 0)) {
                if (strpos($fileString, $code[0]) === 0) {
                    $foundMatch = 1;
                    $settings   = explode(';', $code[1]);
                    $current = explode(';', $currentStyle);
                    for ($y = 0; $y < count($settings); $y++) {    // Process codes.
                        switch ($settings[$y]) {
                            case 0:
                                $current[0] = 0;
                                $current[1] = 0;
                                $current[2] = 0;
                                $current[3] = 0;
                                $current[4] = 0;
                                $current[5] = 0;
                                $current[6] = 37;
                                $current[7] = 40;
                                break;
                            case 1:
                                $current[0] = 1;
                                break;
                            case 2:
                                $current[1] = 1;
                                break;
                            case 4:
                                $current[2] = 1;
                                break;
                            case 5:
                                $current[3] = 1;
                                break;
                            case 7:
                                $current[4] = 1;
                                break;
                            case 8:
                                $current[5] = 1;
                                break;
                            case 30:
                                $current[6] = 30;
                                break;
                            case 31:
                                $current[6] = 31;
                                break;
                            case 32:
                                $current[6] = 32;
                                break;
                            case 33:
                                $current[6] = 33;
                                break;
                            case 34:
                                $current[6] = 34;
                                break;
                            case 35:
                                $current[6] = 35;
                                break;
                            case 36:
                                $current[6] = 36;
                                break;
                            case 37:
                                $current[6] = 37;
                                break;
                            case 40:
                                $current[7] = 40;
                                break;
                            case 41:
                                $current[7] = 41;
                                break;
                            case 42:
                                $current[7] = 42;
                                break;
                            case 43:
                                $current[7] = 43;
                                break;
                            case 44:
                                $current[7] = 44;
                                break;
                            case 45:
                                $current[7] = 45;
                                break;
                            case 46:
                                $current[7] = 46;
                                break;
                            case 47:
                                $current[7] = 47;
                                break;
                        }
                    }
                    $currentStyle = implode(';', $current);
                    $style[$arrayPos] = $currentStyle;
                    $fileString = preg_replace("/\e\[(((\d){1,2};{0,1}){0,6})[m]/", '', $fileString, 1);
                }
            }
        }
    }

    // Combine the array together and return the results.
    $pos = 0;
    $string = '<table><tr><td><pre style="font-family: Courier New; font-size: ' . $fontSize . 'pt; color: #ADAAAD; background-color: #000000";>';
    $lastspan = '';
    $spantemp = 0;
    for ($x = 0; $x < $maxSize; $x++) {
        // Determine the span tag.
        $span = '<span style="';
        $attributes = explode(';', $style[$x]);

        switch ($attributes[6]) {
            case 30:
                if ($attributes[0] == 0) {
                    $span = $span . 'color: #000000;';
                } else {
                    $span = $span . 'color: #525552;';
                }
                break;
            case 31:
                if ($attributes[0] == 0) {
                    $span = $span . 'color: #AD5500;';
                } else {
                    $span = $span . 'color: #FF5552;';
                }
                break;
            case 32:
                if ($attributes[0] == 0) {
                    $span = $span . 'color: #00AA00;';
                } else {
                    $span = $span . 'color: #52FF52;';
                }
                break;
            case 33:
                if ($attributes[0] == 0) {
                    $span = $span . 'color: #ADAA00;';
                } else {
                    $span = $span . 'color: #FFFF52;';
                }
                break;
            case 34:
                if ($attributes[0] == 0) {
                    $span = $span . 'color: #0055AD;';
                } else {
                    $span = $span . 'color: #5255FF;';
                }
                break;
            case 35:
                if ($attributes[0] == 0) {
                    $span = $span . 'color: #AD00AD;';
                } else {
                    $span = $span . 'color: #FF55FF;';
                }
                break;
            case 36:
                if ($attributes[0] == 0) {
                    $span = $span . 'color: #00AAAD;';
                } else {
                    $span = $span . 'color: #52FFFF;';
                }
                break;
            case 37:
                if ($attributes[0] == 0) {
                    $span = $span . 'color: #ADAAAD;';
                } else {
                    $span = $span . 'color: #FFFFFF;';
                }
                break;
        }
        switch ($attributes[7]) {
            case 40:
                // Removed due to being default value.
                //$span = $span . ' background-color: #000000;';
                break;
            case 41:
                $span = $span . ' background-color: #AD5500;';
                break;
            case 42:
                $span = $span . ' background-color: #00AA00;';
                break;
            case 43:
                $span = $span . ' background-color: #ADAA00;';
                break;
            case 44:
                $span = $span . ' background-color: #0055AD;';
                break;
            case 45:
                $span = $span . ' background-color: #AD00AD;';
                break;
            case 46:
                $span = $span . ' background-color: #00AAAD;';
                break;
            case 47:
                $span = $span . ' background-color: #ADAAAD;';
                break;
        }

        $span = $span . '">';

        // This will check to see if the span tag was the same as the last. If
        // so, it will not use it since it just slows down the display and is
        // not needed.
        if ($span != $lastspan) {
            $lastspan = $span;
            if ($spantemp == 1) {
                $span = '</span>' . $span;
            }
            //$string = $string . $span . HtmlSpace($content[$x]);
            $string = $string . $span . $content[$x];
            $pos++;
        } else {
            //$string = $string . HtmlSpace($content[$x]);
            $string = $string . $content[$x];
            $spantemp = 1;
            $pos++;
        }

        if ($pos > 79) {
            $string = $string . "\n";
            $pos = 0;
        }
    }

    // Truncate blank rows.
    if ($truncate) {
        $string = rtrim($string);
    }

    if ($spantemp == 1) {
        $string = $string . '</span>';
    }
    $string = $string . '</pre></tr></td></table>';

    return $string;                         // Return the HTML output.
}


/**
* Va-Functions.inc.php :: ProgramTime()
*
* This function simply returns the current timestamp in microseconds.
*
* @return integer
*/
function ProgramTime() {
    $time = microtime();
    $time = explode(' ', $time);
    return $time[0] + $time[1];
}


/**
* Va-Functions.inc.php :: PutChar()
*
* This function will put the passed integer into the file string at the given
* position. This is used for writing characters, or one byte integers.
*
* @param string $fileString
* @param integer $int
* @param integer $position
* @return string
*/
function PutChar($fileString, $int, $position) {
    if (!is_string($fileString)) {
        $fileString = (string)  $fileString;
    }
    if (!is_int($int)) {
        $int        = (integer) $int;
    }
    if (!is_int($position)) {
        $position   = (integer) $position;
    }

    if (strlen($fileString) < $position) {
       $fileString = $fileString . str_replace('*', chr(0), str_pad('', $position, '*'));
    }

    $position--;

    $int = MakeChar($int);
    $fileString = substr_replace($fileString, $int, $position, 1);

    return $fileString;                     // Return the new string.
}


/**
* Va-Functions.inc.php :: PutInteger()
*
* This function will put the passed integer into the file string at the given
* position. This is used for writing two byte integers. The default maximum
* value can be 32767, but if you set $unsigned to 1 then the maximum is raised to
* 65535 as long as it is a positive value.
*
* @param string $fileString
* @param integer $int
* @param integer $position
* @param boolean $unsigned
* @return string
*/
function PutInteger($fileString, $int, $position, $unsigned = 0) {
    if (!is_string($fileString)) {
        $fileString = (string)  $fileString;
    }
    if (!is_int($int)) {
        $int        = (integer) $int;
    }
    if (!is_int($position)) {
        $position   = (integer) $position;
    }
    if (!is_bool($unsigned)) {
        $unsigned   = (boolean) $unsigned;
    }

    if (strlen($fileString) < $position) {
       $fileString = $fileString . str_replace('*', chr(0), str_pad('', $position, '*'));
    }

    $position--;

    $int = MakeInteger($int, $unsigned);
    $fileString = substr_replace($fileString, $int, $position, 2);

    return $fileString;                     // Return the new string.
}


/**
* Va-Functions.inc.php :: PutLongInt()
*
* This function will put the passed integer into the file string at the given
* position. This is used for writing four byte integers. The maximum value
* can be 2147483647, but if you set $unsigned to 1, then the maximum is raise to
* 4294967295 as long as it is a positive value.
*
* @param string $fileString
* @param integer $int
* @param integer $position
* @param boolean $unsigned
* @return string
*/
function PutLongInt($fileString, $int, $position, $unsigned = 0) {
    if (!is_string($fileString)) {
        $fileString = (string)  $fileString;
    }
    if (!is_int($int)) {
        $int        = (integer) $int;
    }
    if (!is_int($position)) {
        $position   = (integer) $position;
    }
    if (!is_bool($unsigned)) {
        $unsigned   = (boolean) $unsigned;
    }

    if (strlen($fileString) < $position) {
       $fileString = $fileString . str_replace('*', chr(0), str_pad('', $position, '*'));
    }

    $position--;

    $int = MakeLongInt($int, $unsigned);
    $fileString = substr_replace($fileString, $int, $position, 4);

    return $fileString;                     // Return the new string.
}


/**
* Va-Functions.inc.php :: PutSingle()
*
* This function will put the passed single precision value into the file string
* at the given position.
*
* @param string $fileString
* @param integer $float
* @param integer $position
* @return string
*/
function PutSingle($fileString, $float, $position) {
    if (!is_string($fileString)) {
        $fileString = (string)  $fileString;
    }
    if (!is_float($float)) {
        $float      = (float)   $float;
    }
    if (!is_int($position)) {
        $position   = (integer) $position;
    }

    if (strlen($fileString) < $position) {
       $fileString = $fileString . str_replace('*', chr(0), str_pad('', $position, '*'));
    }

    $position--;

    $float = MakeSingle($float);
    $fileString = substr_replace($fileString, $float, $position, 4);

    return $fileString;                     // Return the new string.
}


/**
* Va-Functions.inc.php :: PutString()
*
* This function will put the passed string into the file string at the given
* position.
*
* @param string $fileString
* @param string $string
* @param integer $position
* @param integer $length
* @return string
*/
function PutString($fileString, $string, $position, $length) {
    if (!is_string($fileString)) {
        $fileString = (string)  $fileString;
    }
    if (!is_string($string)) {
        $string     = (string)  $string;
    }
    if (!is_int($position)) {
        $position   = (integer) $position;
    }
    if (!is_int($length)) {
        $length     = (integer) $length;
    }
    
    if (strlen($fileString) < $position) {
       $fileString = $fileString . str_replace('*', chr(0), str_pad('', $position, '*'));
    }

    $position--;

    if (strlen($string) < $length) {
        $string = str_pad($string, $length, chr(32));
    } else {
        $string = substr($string, 0, $length);
    }
    $fileString = substr_replace($fileString, $string, $position, $length);
    
    return $fileString;                     // Return the new string.
}


/**
* Va-Functions.inc.php :: ReadAt2kConfig()
*
* This function will read an AT2k script configuration file, which usually is
* stored in the BBS's TXT directory.
*
* The configuration file has comments designated by a semicolon, and values are
* positioned based (they must appear in the file in a certain order and are not
* assigned to variables).
*
* If trim is set to FALSE, then it will not trim the line of any spaces.
*
* @param string $configFile
* @return array
*/
function ReadAt2kConfig($configFile, $trim = 1) {
    if (!is_string($configFile)) {
        $configFile = (string) $configFile;
    }
    if (!is_bool($trim)) {
        $trim = (boolean) $trim;
    }

    // If the configfile is missing, send an error.
    if (!is_file($configFile)) {
        return FALSE;
    }

    // Read the configfile into an array.
    $file = file($configFile);              // Read the configfile.
    $cnt  = 1;

    for ($x = 0; $x < count($file); $x++) {
        $line    = trim($file[$x]);         // Read a line.
        $comment = trim(substr($line, 0, 1));
        if (($comment != ';') && ($line != '')) {
            if ($trim == TRUE) {
                $var[$cnt] = $line;         // Add the line into the array.
            } else {
                $var[$cnt] = $file[$x];     // Add non trimmed line into the array.
            }
            $cnt++;
        }
    }

    return $var;
}


/**
* Va-Functions.inc.php :: ReadCfgFile()
*
* This function will read the entire file specified into one variable.
*
* @param string $file
* @return string
*/
function ReadCfgFile($file) {
    if (!is_string($file)) {
        $file = (string) $file;
    }

    // Check to see if the file exists, if not return nothing.
    if (!is_file($file)) {
        return '';                          // If file is missing, then it is blank.
    }
    if ((is_file($file)) && (!is_readable($file))) {
        return FALSE;                       // If file exists but cannot be read, return FALSE.
    }

    $fileString = '';

    if (version_compare(phpversion(), '4.3.0', '>=')) {
        $fileString = @file_get_contents($file);
    } else if (is_file($file)) {
        $fp = fopen($file, 'rb');
        if (flock($fp, LOCK_SH | LOCK_NB)) {
            $fileString = fread($fp, filesize($file));
            flock($fp, LOCK_UN);
        }
        fclose($fp);
    }

    return $fileString;                     // Return the file string.
}


/**
* Va-Functions.inc.php :: ReadConfig()
*
* This function will read a configuration file and place its variables into
* an array.
*
* The type of configuration file has comments designated by a semicolon, and
* variables are assigned like 'variable='value''.
*
* @param string $configFile
* @return array
*/
function ReadConfig($configFile) {
    if (!is_string($configFile)) {
        $configFile = (string) $configFile;
    }

    // If the configfile is missing, return the default values.
    if (!is_file($configFile)) {
        return FALSE;
    }

    // Read the configfile into an array.
    $file = file($configFile);              // Read the configfile.
    $cnt  = 1;

    for ($x = 0; $x < count($file); $x++) {
        $line    = trim($file[$x]);         // Read a line.
        $comment = trim(substr($line, 0, 1));
        if (($comment != ';') && ($line != '')) {
            $var[$cnt] = $line;             // Add the line into the array.
            $cnt++;
        }
    }

    // The configuration is now in an array, so lets set the proper variables
    // with the information.

    for ($y = 1; $y != $cnt; $y++) {                            // Examine the array and set new values.
        $pos   = strpos($var[$y], '=');                         // Determine the position of the equal symbol.
        $desc  = trim(strtolower(substr($var[$y], 0, $pos)));   // Determine the variable description.
        $value = trim(substr($var[$y], $pos + 1));              // Determine the variable value.
        $value = str_replace(' ', '', $value);                  // Remove any spaces.
        $config[$desc] = strtolower($value);
    }

    return $config;                         // Return the configuration array.
}


/**
* Va-Functions.inc.php :: ReadID3Info()
*
* This function will read the ID3 information from a MP3 file.
*
* @param [type] $file
* @return
*/
function ReadID3Info($file) {
    if (!is_string($file)) {
        $file = (string) $file;
    }

    // Set default information.
    $id3['tag']      = '';
    $id3['songname'] = '';
    $id3['artist']   = '';
    $id3['album']    = '';
    $id3['year']     = '';
    $id3['comment']  = '';
    $id3['genre']    = '';

    if (!is_file($file)) {
        return $id3;
    }

    $length = filesize($file);
    if ($length < 128) {
        return $id3;
    }

    $fp = fopen($file, 'rb');
    fseek($fp, $length - 128);              // Set pointer for last 128 bytes.
    $fileString = fread($fp, 128);          // Read the last 128 bytes.
    fclose($fp);

    $id3['tag'] = GetString($fileString, 1, 3);
    if ($id3['tag'] == 'TAG') {
        $id3['songname'] = GetString($fileString, 4, 30);
        $id3['artist']   = GetString($fileString, 34, 30);
        $id3['album']    = GetString($fileString, 64, 30);
        $id3['year']     = GetString($fileString, 94, 4);
        $id3['comment']  = GetString($fileString, 98, 30);
        $genre           = GetChar($fileString, 128);
        switch ($genre) {
            case 0:     $id3['genre'] = 'Blues';                break;
            case 1:     $id3['genre'] = 'Classic Rock';         break;
            case 2:     $id3['genre'] = 'Country';              break;
            case 3:     $id3['genre'] = 'Dance';                break;
            case 4:     $id3['genre'] = 'Disco';                break;
            case 5:     $id3['genre'] = 'Funk';                 break;
            case 6:     $id3['genre'] = 'Grunge';               break;
            case 7:     $id3['genre'] = 'Hip-Hop';              break;
            case 8:     $id3['genre'] = 'Jazz';                 break;
            case 9:     $id3['genre'] = 'Metal';                break;
            case 10:    $id3['genre'] = 'New Age';              break;
            case 11:    $id3['genre'] = 'Oldies';               break;
            case 12:    $id3['genre'] = 'Other';                break;
            case 13:    $id3['genre'] = 'Pop';                  break;
            case 14:    $id3['genre'] = 'R&B';                  break;
            case 15:    $id3['genre'] = 'Rap';                  break;
            case 16:    $id3['genre'] = 'Reggae';               break;
            case 17:    $id3['genre'] = 'Rock';                 break;
            case 18:    $id3['genre'] = 'Techno';               break;
            case 19:    $id3['genre'] = 'Industrial';           break;
            case 20:    $id3['genre'] = 'Alternative';          break;
            case 21:    $id3['genre'] = 'Ska';                  break;
            case 22:    $id3['genre'] = 'Death Metal';          break;
            case 23:    $id3['genre'] = 'Pranks';               break;
            case 24:    $id3['genre'] = 'Soundtrack';           break;
            case 25:    $id3['genre'] = 'Euro-Techno';          break;
            case 26:    $id3['genre'] = 'Ambient';              break;
            case 27:    $id3['genre'] = 'Trip-hop';             break;
            case 28:    $id3['genre'] = 'Vocal';                break;
            case 29:    $id3['genre'] = 'Jazz+Funk';            break;
            case 30:    $id3['genre'] = 'Fusion';               break;
            case 31:    $id3['genre'] = 'Trance';               break;
            case 32:    $id3['genre'] = 'Classical';            break;
            case 33:    $id3['genre'] = 'Instrumental';         break;
            case 34:    $id3['genre'] = 'Acid';                 break;
            case 35:    $id3['genre'] = 'House';                break;
            case 36:    $id3['genre'] = 'Game';                 break;
            case 37:    $id3['genre'] = 'Sound Clip';           break;
            case 38:    $id3['genre'] = 'Gospel';               break;
            case 39:    $id3['genre'] = 'Noise';                break;
            case 40:    $id3['genre'] = 'AlternRock';           break;
            case 41:    $id3['genre'] = 'Bass';                 break;
            case 42:    $id3['genre'] = 'Soul';                 break;
            case 43:    $id3['genre'] = 'Punk';                 break;
            case 44:    $id3['genre'] = 'Space';                break;
            case 45:    $id3['genre'] = 'Meditative';           break;
            case 46:    $id3['genre'] = 'Instrumental Pop';     break;
            case 47:    $id3['genre'] = 'Instrumental Rock';    break;
            case 48:    $id3['genre'] = 'Ethnic';               break;
            case 49:    $id3['genre'] = 'Gothic';               break;
            case 50:    $id3['genre'] = 'Darkwave';             break;
            case 51:    $id3['genre'] = 'Techno-Industrial';    break;
            case 52:    $id3['genre'] = 'Electronic';           break;
            case 53:    $id3['genre'] = 'Pop-Folk';             break;
            case 54:    $id3['genre'] = 'Eurodance';            break;
            case 55:    $id3['genre'] = 'Dream';                break;
            case 56:    $id3['genre'] = 'Southern Rock';        break;
            case 57:    $id3['genre'] = 'Comedy';               break;
            case 58:    $id3['genre'] = 'Cult';                 break;
            case 59:    $id3['genre'] = 'Gansta';               break;
            case 60:    $id3['genre'] = 'Top 40';               break;
            case 61:    $id3['genre'] = 'Christian Rap';        break;
            case 62:    $id3['genre'] = 'Pop/Funk';             break;
            case 63:    $id3['genre'] = 'Jungle';               break;
            case 64:    $id3['genre'] = 'Native American';      break;
            case 65:    $id3['genre'] = 'Cabaret';              break;
            case 66:    $id3['genre'] = 'New Wave';             break;
            case 67:    $id3['genre'] = 'Psychadelic';          break;
            case 68:    $id3['genre'] = 'Rave';                 break;
            case 69:    $id3['genre'] = 'Showtunes';            break;
            case 70:    $id3['genre'] = 'Trailer';              break;
            case 71:    $id3['genre'] = 'Lo-Fi';                break;
            case 72:    $id3['genre'] = 'Tribal';               break;
            case 73:    $id3['genre'] = 'Acid Punk';            break;
            case 74:    $id3['genre'] = 'Acid Jazz';            break;
            case 75:    $id3['genre'] = 'Polka';                break;
            case 76:    $id3['genre'] = 'Retro';                break;
            case 77:    $id3['genre'] = 'Musical';              break;
            case 78:    $id3['genre'] = 'Rock & Roll';          break;
            case 79:    $id3['genre'] = 'Hard Rock';            break;
            case 80:    $id3['genre'] = 'Folk';                 break;
            case 81:    $id3['genre'] = 'Folk-Rock';            break;
            case 82:    $id3['genre'] = 'National Rock';        break;
            case 83:    $id3['genre'] = 'Swing';                break;
            case 84:    $id3['genre'] = 'Fast Fusion';          break;
            case 85:    $id3['genre'] = 'Bebob';                break;
            case 86:    $id3['genre'] = 'Latin';                break;
            case 87:    $id3['genre'] = 'Revival';              break;
            case 88:    $id3['genre'] = 'Celtic';               break;
            case 89:    $id3['genre'] = 'Bluegrass';            break;
            case 90:    $id3['genre'] = 'Avantgarde';           break;
            case 91:    $id3['genre'] = 'Gothic Rock';          break;
            case 92:    $id3['genre'] = 'Progressive Rock';     break;
            case 93:    $id3['genre'] = 'Psychadelic Rock';     break;
            case 94:    $id3['genre'] = 'Symphonic Rock';       break;
            case 95:    $id3['gnere'] = 'Slow Rock';            break;
            case 96:    $id3['genre'] = 'Big Band';             break;
            case 97:    $id3['genre'] = 'Chorus';               break;
            case 98:    $id3['genre'] = 'Easy Listening';       break;
            case 99:    $id3['genre'] = 'Acoustic';             break;
            case 100:   $id3['genre'] = 'Humour';               break;
            case 101:   $id3['genre'] = 'Speech';               break;
            case 102:   $id3['genre'] = 'Chanson';              break;
            case 103:   $id3['genre'] = 'Opera';                break;
            case 104:   $id3['genre'] = 'Chamber Music';        break;
            case 105:   $id3['genre'] = 'Sonata';               break;
            case 106:   $id3['genre'] = 'Symphony';             break;
            case 107:   $id3['genre'] = 'Booty Bass';           break;
            case 108:   $id3['genre'] = 'Primus';               break;
            case 109:   $id3['genre'] = 'Porn Groove';          break;
            case 110:   $id3['genre'] = 'Satire';               break;
            case 111:   $id3['genre'] = 'Slow Jam';             break;
            case 112:   $id3['genre'] = 'Club';                 break;
            case 113:   $id3['genre'] = 'Tango';                break;
            case 114:   $id3['genre'] = 'Samba';                break;
            case 115:   $id3['genre'] = 'Folklore';             break;
            case 116:   $id3['genre'] = 'Ballad';               break;
            case 117:   $id3['genre'] = 'Power Ballad';         break;
            case 118:   $id3['genre'] = 'Rhythmic Sould';       break;
            case 119:   $id3['genre'] = 'Freestyle';            break;
            case 120:   $id3['genre'] = 'Duet';                 break;
            case 121:   $id3['genre'] = 'Punk Rock';            break;
            case 122:   $id3['genre'] = 'Drum Solo';            break;
            case 123:   $id3['genre'] = 'A Capella';            break;
            case 124:   $id3['genre'] = 'Euro-House';           break;
            case 125:   $id3['genre'] = 'Dance Hall';           break;
            default:    $id3['genre'] = 'Unknown';              break;
        }
    }

    return $id3;                            // Return the tag information.
}


/**
* Va-Funtions.inc.php :: RedirectUser()
*
* This function will redirect the user to the specified URL. It uses a primary
* base of JavaScript with a backup meta tag for those browsers that do not
* support JavaScript. Note that when this is called, the current page will
* exit and the redirection will take place. You may set delay the redirection
* a specific number of seconds by using the optional $delay variable.
*
* With WML you may specify additional variables to unset during the redirection,
* in the format of RedirectUser($url, $delay, $var1, $value1, $var2, $value2..).
*
* @param string $url
* @param integer $delay
*/
function RedirectUser($url, $delay = 0) {
    if (!is_string($url)) {
        $url   = (string)  $url;
    }
    if (!is_int($delay)) {
        $delay = (integer) $delay;
    }

    global $includeDir;

    // Check whether the browser/gateway says it accepts WML.
    if ((isset($_SERVER['HTTP_ACCEPT'])) &&
        ((strpos(strtolower($_SERVER['HTTP_ACCEPT']), 'vnd.wap.wml') !== FALSE))) {
        $br = 'WML';
    } else {
        if (isset($_SERVER['HTTP_USER_AGENT'])) {
            $browser = substr(trim($_SERVER['HTTP_USER_AGENT']), 0, 4);
        } else {
            $browser = '';
        }
        if ($browser == 'Noki' ||           // Nokia phones and emulators
            $browser == 'Eric' ||           // Ericsson WAP phones and emulators
            $browser == 'WapI' ||           // Ericsson WapIDE 2.0
            $browser == 'MC21' ||           // Ericsson MC218
            $browser == 'AUR ' ||           // Ericsson R320
            $browser == 'R380' ||           // Ericsson R380
            $browser == 'UP.B' ||           // UP.Browser
            $browser == 'WinW' ||           // WinWAP browser
            $browser == 'UPG1' ||           // UP.SDK 4.0
            $browser == 'upsi' ||           // another kind of UP.Browser ??
            $browser == 'QWAP' ||           // unknown QWAPPER browser
            $browser == 'Jigs' ||           // unknown JigSaw browser
            $browser == 'Java' ||           // unknown Java based browser
            $browser == 'Alca' ||           // unknown Alcatel-BE3 browser (UP based?)
            $browser == 'MITS' ||           // unknown Mitsubishi browser
            $browser == 'MOT-' ||           // unknown browser (UP based?)
            $browser == 'My S' ||           // unknown Ericsson devkit browser ?
            $browser == 'WAPJ' ||           // Virtual WAPJAG www.wapjag.de
            $browser == 'fetc' ||           // fetchpage.cgi Perl script from www.wapcab.de
            $browser == 'ALAV' ||           // yet another unknown UP based browser ?
            $browser == 'Wapa')             // another unknown browser (Web based "Wapalyzer"?)
        {
            $br = 'WML';
        } else {
            $br = 'HTML';
        }
    }

    if (ob_get_length()) {
        ob_clean();                         // Flush the current buffer.
    }
    session_write_close();                  // Stop session writing.

    if ((!headers_sent()) &&
        (func_num_args() <= 2) &&
        ((strpos($url, 'http://') !== FALSE) ||
         (strpos($url, 'https://') !== FALSE)) &&
        (strpos(strtoupper($_SERVER['SERVER_SOFTWARE']), 'IIS') === FALSE)) {
        if ($delay > 0) {
            sleep($delay);
        }
        header('Location: ' . $url);
    } else {
        if ($br == 'WML') {
            if ($delay > 0) {
                sleep($delay);
            }
            header('Content-type: text/vnd.wap.wml');
            print('<?xml version="1.0"?>');
            print('<!DOCTYPE wml PUBLIC "-//WAPFORUM//DTD WML 1.1//EN" "http://www.wapforum.org/DTD/wml_1.1.xml">');

            // Determine if there is a query string to break up.
            $query = parse_url($url);
            if ((isset($query['query'])) && ($query['query'] != '')) {
                $pos = strpos($url, $query['query']);
                $url = substr($url, 0, $pos - 1);
                parse_str($query['query'], $query);
            } else {
                $query = FALSE;
            }
?>

<wml>
    <card name="Redirect">
        <onevent type="onenterforward">
            <go href="<?php print $url; ?>" method="get">
<?php
            if ($query) {
                foreach ($query as $key => $value) {
?>
                <postfield name="<?php print $key; ?>" value="<?php print $value; ?>" />
<?php
                }
            }

            for ($x = 2; $x < func_num_args(); $x += 2) {
?>
                <setvar name="<?php print func_get_arg($x); ?>" value="<?php print func_get_arg($x + 1); ?>" />
<?php
            }
?>
            </go>
        </onevent>
        <onevent type="onenterbackward">
            <go href="<?php print $url; ?>" method="get">
<?php
            if ($query) {
                foreach ($query as $key => $value) {
?>
                <postfield name="<?php print $key; ?>" value="<?php print $value; ?>" />
<?php
                }
            }

            for ($x = 2; $x < func_num_args(); $x += 2) {
?>
                <setvar name="<?php print func_get_arg($x); ?>" value="<?php print func_get_arg($x + 1); ?>" />
<?php
            }
?>
            </go>
        </onevent>
    </card>
</wml>

<?php
        } else {
?>

<html>
<head>
    <meta http-equiv="Refresh" content="<?php print $delay; ?>; url=<?php print $url; ?>" />
</head>
<body>
    <noscript>
        <b>Your browser does not have JavaScript enabled.</b><br />
        Enable JavaScript for a better experience while using this site.<br />
        Click <a href="<?php print $url; ?>">here</a> to continue.
    </noscript>
</body>
</html>

<?php
        }
    }

    die();
}


/**
* Va-Functions.inc.php :: SendFile()
*
* This function will send the specified file to the user using the fpassthru()
* function.
*
* @param string $file
* @return mixed
*/
function SendFile($file) {
    if (!is_string($file)) {
        $file = (string) $file;
    }

    if (!is_file($file)) {
        return FALSE;
    }

    if (ob_get_length()) {
        ob_end_clean();                     // Turn off buffering.
    }
    session_write_close();                  // Stop session writing.

    $status     = FALSE;                    // Set the status to default to false.
    $filename   = basename($file);          // Get the filename.
    $filesize   = filesize($file);          // Get the filesize.
    $filesizemb = intval($filesize / 1048576);

    // Determine MIME type.
    $pos = strrpos($filename, '.');
    $ext = strtolower(substr($filename, $pos));
    switch ($ext) {
        case '.ai':   $mimetype='application/postscript';   $encode='7bit';     break;
        case '.avi':  $mimetype='video/x-msvideo';          $encode='base64';   break;
        case '.doc':  $mimetype='application/msword';       $encode='base64';   break;
        case '.eps':  $mimetype='application/postscript';   $encode='7bit';     break;
        case '.exe':  $mimetype='application/octet-stream'; $encode='base64';   break;
        case '.gif':  $mimetype='image/gif';                $encode='base64';   break;
        case '.htm':  $mimetype='text/html';                $encode='7bit';     break;
        case '.html': $mimetype='text/html';                $encode='7bit';     break;
        case '.jpe':  $mimetype='image/jpeg';               $encode='base64';   break;
        case '.jpeg': $mimetype='image/jpeg';               $encode='base64';   break;
        case '.jpg':  $mimetype='image/jpeg';               $encode='base64';   break;
        case '.mpe':  $mimetype='video/mpeg';               $encode='base64';   break;
        case '.mpeg': $mimetype='video/mpeg';               $encode='base64';   break;
        case '.mov':  $mimetype='video/quicktime';          $encode='base64';   break;
        case '.mpg':  $mimetype='video/mpeg';               $encode='base64';   break;
        case '.pdf':  $mimetype='application/pdf';          $encode='base64';   break;
        case '.ps':   $mimetype='application/postscript';   $encode='7bit';     break;
        case '.qt':   $mimetype='video/quicktime';          $encode='base64';   break;
        case '.rtf':  $mimetype='application/rtf';          $encode='7bit';     break;
        case '.tar':  $mimetype='application/octet-stream'; $encode='base64';   break;
        case '.tif':  $mimetype='image/tiff';               $encode='base64';   break;
        case '.tiff': $mimetype='image/tiff';               $encode='base64';   break;
        case '.txt':  $mimetype='text/plain';               $encode='7bit';     break;
        case '.wav':  $mimetype='audio/x-wav';              $encode='base64';   break;
        case '.zip':  $mimetype='application/zip';          $encode='base64';   break;
        default:      $mimetype='application/octet-stream'; $encode='base64';   break;
    }

    // Set some new time limits and file sizes. (Do not think this is necessary)
    ini_set('max_execution_time', 240);
    ini_set('memory_limit', 4 + $filesizemb . 'M');

    header('Cache-Control: ');
    header('Pragma: ');
    header('Content-type: ' . $mimetype);
    header('Content-Disposition: attachment; filename="' . $filename . '"');
    header('Content-transfer-encoding: ' . $encode);
    header('Content-length: ' . $filesize);

    // Send the file using fread.
    $fp = fopen($file, 'rb');
    while(!feof($fp)) {
        print fread($fp, 4096);
        flush();
    }
    fclose($fp);
    $status = (connection_status() == 0);

    // Send the file using fpassthru.
//    $fp = fopen($file, 'rb');               // Open the file to send.
//    $status = fpassthru($fp);               // Send the file.

    return $status;                         // Return the status.
}


/**
* Va-Functions.inc.php :: SendOneLiner()
*
* This function will send a one-liner to the specified node, or to all active
* nodes. If the userNo is 0, then it will send the message to any user. If
* the node is set to all, it will send it to all active nodes.
*
* @param string $message
* @param integer $userNo
* @param string $node
* @return mixed
*/
function SendOneLiner($message, $userNo = 0, $node = 'All') {
    if (!is_string($message)) {
        $message = (string)  $message;
    }
    if (!is_int($userNo)) {
        $userNo  = (integer) $userNo;
    }
    if (!is_string($node)) {
        $node    = (string)  $node;
    }

    global $VA_SYSPATHS;                                // Include global variable.
    global $includeDir;
    include_once($includeDir . 'va-control.inc.php');   // Include control.dat functions.
    include_once($includeDir . 'va-main.inc.php');      // Include main configuration.
    global $VA_MAIN;
    
    // Truncate the message if too long.
    if (strlen($message) > 75) {
        $message = substr($message, 0, 75);
    }
    // Add coloring.
    $message = chr(3) . '6*** ' . chr(3) . '1' . $message . "\r\n";

    $status = FALSE;
    if (($userNo == 0) && ($node != 'All')) {
        $file = $VA_SYSPATHS['control'] . '/IN' . $node . '.QUE';
        $fp = fopen($file, 'a');            // Open the file.
        flock($fp, LOCK_EX);                // Lock the file.
        $status = fwrite($fp, $message);    // Write the message.
        flock($fp, LOCK_UN);                // Unlock the file.
        fclose($fp);
    } else {
        if ($userNo > 0) {
            for ($x = 0; $x < $VA_MAIN['maxchannelno']; $x++) {
                $channel = GetControl($x);
                if ($channel['userno'] == $userNo) {
                    $file = $VA_SYSPATHS['control'] . '/IN' . $x . '.QUE';
                    $fp = fopen($file, 'a');            // Open the file.
                    flock($fp, LOCK_EX);                // Lock the file.
                    $status = fwrite($fp, $message);    // Write the message.
                    flock($fp, LOCK_UN);                // Unlock the file.
                    fclose($fp);
                }
            }
        } else {
            for ($x = 0; $x < $VA_MAIN['maxchannelno']; $x++) {
                $channel = GetControl($x);
                if ($channel['userno'] > 0) {
                    $file = $VA_SYSPATHS['control'] . '/IN' . $x . '.QUE';
                    $fp = fopen($file, 'a');            // Open the file.
                    flock($fp, LOCK_EX);                // Lock the file.
                    $status = fwrite($fp, $message);    // Write the message.
                    flock($fp, LOCK_UN);                // Unlock the file.
                    fclose($fp);
                }
            }
        }
    }

    return $status;                         // Return the status.
}


/**
* Va-Functions.inc.php :: StripANSICodes()
*
* This function will strip all standard ANSI codes from the passed string.
*
* @param string $string
* @return string
*/
function StripANSICodes($string) {
    if (!is_string($string)) {
        $string = (string) $string;
    }

    // Check to see if the variable was passed, if not then return the string.
    if (trim($string) == '') {
        return $string;
    }

    $pattern = array(
                    "/\e\[(\d){0,3}[A|B|C|D]/",
                    "/\e\[((\?*?(\d){1,2})|((\d){1,2};{0,1})){0,2}[f|R|h|H|l|y]/",
                    "/\e\[2J/",
                    "/\e\[(((\d){1,2};{0,1}){0,4})[m|p]/",
                    "/\e\[6[n]/",
                    "/\e\[((\d){1,2};{0,1}){0,3}[@|L|M|P]/",
                    "/\e\[[K|s|u]/"
                    );

    return preg_replace($pattern, '', $string); // Return the new string.
}


/**
* Va-Functions.inc.php :: StripBBSCodes()
*
* This routine will take $string and remove any heart codes within the string.
* It also removes diamond codes.
*
* @param string $string
* @return string
*/
function StripBBSCodes($string) {
    if (!is_string($string)) {
        $string = (string) $string;
    }

    // Check to see if the variable was passed, if not then return the string.
    if (trim($string) == '') {
        return $string;
    }

    // Search for SBBS codes.
    $string = preg_replace("/\x01\S/", '', $string);

    // Search for hearts.
    $string = preg_replace("/\x03\S/", '', $string);

    // Search the file for diamonds.
    $string = preg_replace("/\x04\S/", '', $string);

    // Search for pipe codes.
    $string = preg_replace("/\|(\d){2,2}/", '', $string);

    // Search for PcBoard codes.
    $string = preg_replace("/@X\S\S/", '', $string);

    // Search for Wildcat codes.
    $string = preg_replace("/@\S\S@/", '', $string);

    // Search for Celerity codes.
    $string = preg_replace("/\|[k|b|g|c|r|m|y|w|d|B|G|C|R|M|Y|W|S]/", '', $string);

    return $string;                         // Return the new string.
}


/**
* Va-Functions.inc.php :: StripRouting()
*
* This function will take $string and remove the lines that begin with a
* diamond code.
*
* @param string $string
* @return string
*/
function StripRouting($string) {
    if (!is_string($string)) {
        $string = (string) $string;
    }

    // Check to see if the variable was passed, if not then return the string.
    if (trim($string) == '') {
        return $string;
    }

    // Search for diamonds.
    for ($position = 0; is_int($position);) {
        $position = strpos($string, chr(4), 0);
        if (is_int($position)) {
            $endPosition = strpos($string, "\n", $position);
            if ($endPosition === FALSE) {
                $endPosition = strlen($string);
            }
            $string = substr_replace($string, '', $position, ($endPosition - $position));
        }
    }

    return rtrim($string);                  // Return the new string.
}


/**
* Va-Functions.inc.php :: UnHtmlSpecialChars()
*
* Reverses the htmlspecialchars() function.
*
* @param string $string
* @return string
*/
function UnHtmlSpecialChars($string) {
    if (!is_string($string)) {
        $string = (string) $string;
    }

    $string = str_replace('&amp;', '&', $string);
    $string = str_replace('&#039;', '\'', $string);
    $string = str_replace('&quot;', '"', $string);
    $string = str_replace('&lt;', '<', $string);
    $string = str_replace('&gt;', '>', $string);
   
    return $string;
}


/**
* Va-Functions.inc.php :: UpdateWhosOnline()
*
* This function will maintain the who's online listing for VADV-PHP. It is
* automatically called by va-functions.inc.php. The variable action records
* what action the user is taking.
*
* You can set the timeout, which will change the user's time entry. If you set
* it more than 900, then the user will stay on the who's list longer, or if you
* set it below 900, then the user will stay on the list a shorter period of time.
*
* If id is set to true, then it will assign a new id to the user.
*
* @return array
*/
function UpdateWhosOnline($action = '', $timeout = 900, $resetId = 0) {
    if (!is_string($action)) {
        $action = (string) $action;
    }
    if (!is_int($timeout)) {
        $timeout = (integer) $timeout;
    }
    if (!is_bool($resetId)) {
        $resetId = (boolean) $resetId;
    }

    global $VA_SYSPATHS;                    // Include global variable.
    global $includeDir;
    global $USER;

    include_once($includeDir . '/va-stats.inc.php');

    $fileOffset = 159;                      // Offset size.

    $file = $VA_SYSPATHS['data'] . '/WEBCALLER.LST';    // Database filename.
    // If the filesize doesn't match up to the file offset, delete it and
    // start over.
    if ((filesize($file) % $fileOffset) != 0) {
        @unlink ($file);
    }
    $fileString = ReadCfgFile($file);       // Read the file.

    // Read through the file looking for the current user's IP, and also
    // expired users. Users expire after 15 mins (900s).
    $user   = array();
    $online = array();
    $offset = 0;
    for ($x = 1; $offset < strlen($fileString); $x++) {
        $user['session'] = trim(GetString($fileString, 1 + $offset, 32));
        $user['ip']      = trim(GetString($fileString, 33 + $offset, 15));
        $user['time']    = (integer) GetString($fileString, 48 + $offset, 12);
        $user['userno']  = (integer) GetInteger($fileString, 60 + $offset);
        $user['handle']  = trim(GetString($fileString, 62 + $offset, 40));
        $user['action']  = trim(GetString($fileString, 102 + $offset, 24));
        $user['id']      = trim(GetString($fileString, 126 + $offset, 32));
        $user['ssl']     = (integer) GetInteger($fileString, 158 + $offset);

        // Compare values to see who to remove.
        if (($user['session'] == trim(session_id())) ||
            (time() - $user['time'] > 900) ||
            (($USER['userno'] > 0) &&
             (($user['userno'] == $USER['userno']) ||
              ($user['handle'] == $USER['handle'])))) {
            if (($user['id'] != '') && (!$resetId)) {
                $id = $user['id'];
            }
            $fileString = substr_replace($fileString, '', $offset, $fileOffset);
            $x--;
        } else {
            // Write to the new fileString...
            $offset = $fileOffset * $x;
            $online[$x]['session'] = trim($user['session']);
            $online[$x]['ip']      = trim($user['ip']);
            $online[$x]['time']    = trim($user['time']);
            $online[$x]['userno']  = trim($user['userno']);
            $online[$x]['handle']  = trim($user['handle']);
            $online[$x]['action']  = trim($user['action']);
            $online[$x]['id']      = trim($user['id']);
            $online[$x]['ssl']     = trim($user['ssl']);
        }
    }

    // Add the current user's entry to the end if not a web spider.
    if (!DetectSpider()) {
        // Determine time difference
        $timeout = $timeout - 900;

        $sess   = str_pad(session_id(), 32);
        $ip     = str_pad($_SERVER['REMOTE_ADDR'], 15);
        $time   = str_pad(time() + $timeout, 12);
        $userno = MakeInteger($USER['userno']);
        $handle = str_pad($USER['handle'], 40);
        $action = str_pad(trim($action), 24);
        if (!isset($id)) {
            $id = str_pad(md5(rand(-32767, 32767)), 32);
        } else {
            $id = str_pad($id, 32);
        }
        if (IsHttps()) {
            $ssl = 1;
        } else {
            $ssl = 0;
        }
        $sslString = MakeInteger($ssl);

        $fileString = $fileString . $sess . $ip . $time . $userno . $handle . $action . $id . $sslString;

        $online[$x]['session'] = trim($sess);
        $online[$x]['ip']      = trim($ip);
        $online[$x]['time']    = trim($time);
        $online[$x]['userno']  = trim($USER['userno']);
        $online[$x]['handle']  = trim($USER['handle']);
        $online[$x]['action']  = trim($action);
        $online[$x]['id']      = trim($id);
        $online[$x]['ssl']     = trim($ssl);
    } else {
        $x--;
    }

    // Write the new file.
    WriteCfgFile($file, $fileString);

    // Organize the array alphabetically by handle.
    if (count($online) > 0) {
        foreach ($online as $row) {
            $sortarr[] = $row['handle'];
        }
        array_multisort($sortarr, $online);
        array_unshift($online, $x);
    } else {
        $online[0] = 0;
    }

    $fileString = ReadStats();              // Read the stats.
    $stats = GetStats($fileString);
    // Update max users online.
    if ($online[0] > $stats['webmaxusers']) {
        $stats['webmaxusers'] = $online[0];
        $fileString = PutStats($fileString, $stats);
        WriteStats($fileString);            // Write the new file.
    }

    return $online;                         // Return the online users.
}


/**
* Va-Functions.inc.php :: UserDate()
*
* This function is just like the PHP date() function except it will factor in
* the user's timezone.
*
* @param [type] $format
* @return
*/
function UserDate($format) {
    global $USER_SETTINGS;
    $timeStamp = time() - ((integer) date("O") / 100 * 60 * 60) + ($USER_SETTINGS['timezone'] / 100 * 60 * 60);
    return date($format, $timeStamp);
}


/**
* Va-Functions.inc.php :: VA_Die()
*
* This function replaces the php die() command. It will display an error message
* for five seconds, and then if the directory contains an index.htm, index.html
* or index.php file, it will redirect to it.
*
* @param string $message
* @param integer $delay
*/
function VA_Die($message, $delay = 5) {
    if (!is_string($message)) {
        $message = (string)  $message;
    }
    if (!is_int($delay)) {
        $delay   = (integer) $delay;
    }

    $index = '';
    if (is_file('index.php')) {
        $index = 'index.php';
    } else if (is_file('index.htm')) {
        $index = 'index.htm';
    } else if (is_file('index.html')) {
        $index = 'index.html';
    }

    if ($index != '') {
        print $message;
        ob_flush();
        RedirectUser($index, $delay);
    } else {
        die($message);
    }
}


/**
* Va-Functions.inc.php :: VA_Error()
*
* This function will return the specified error message.
*
* @param integer $errNum
* @return string
*/
function VA_Error($errNum) {
    if (!is_int($errNum)) {
        $errNum = (integer) $errNum;
    }

    switch ($errNum) {
        case 0:
            $errMsg = 'File not found.';
            break;
        case 1:
            $errMsg = 'Variable is not proper type.';
            break;
        case 2:
            $errMsg = 'Value is out of range.';
            break;
        case 3:
            $errMsg = 'Invalid Configuration.';
            break;
        case 4:
            $errMsg = 'Unknown Error.';
            break;
    }
    
    return $errMsg;                         // Return the error message.
}


/**
* Va-Functions.inc.php :: VA_LoginError()
*
* This function will display a message stating that the user needs to login to
* access a certain area and give a link to the generic login page.
*
*/
function VA_LoginError() {
    global $CONFIG;                         // Include global variable.
    global $rootDir;
    global $includeDir;

    require_once($includeDir. 'html_header.php');
?>
<html>

<head>
<title>Login Authentication Error</title>
<meta name="copyright" content="Copyright  2003, Steve Winn" />
<style type="text/css">
<!--
a.infoBarBody:link    { font-family: Verdana, Arial, Helvetica; font-size: 8pt;  color: #FFFFFF; font-weight:bold }
a.infoBarBody:visited { font-family: Verdana, Arial, Helvetica; font-size: 8pt;  color: #FFFFFF; font-weight:bold }
a.navBar:link         { font-family: Verdana, Arial, Helvetica; font-size: 8pt;  color: #FFFFFF; font-weight:bold }
a.navBar:visited      { font-family: Verdana, Arial, Helvetica; font-size: 8pt;  color: #FFFFFF; font-weight:bold }
a.return:link         { font-family: Verdana, Arial, Helvetica; font-size: 10pt; color: #FFFF00; font-weight:bold }
a.return:visited      { font-family: Verdana, Arial, Helvetica; font-size: 10pt; color: #FFFF00; font-weight:bold }
.infoBarBody          { font-family: Verdana, Arial, Helvetica; font-size: 8pt;  color: #FFFFFF; }
.navbar               { font-family: Verdana, Arial, Helvetica; font-size: 8pt;  color: #FFFFFF; font-weight:bold }
.return               { font-family: Verdana, Arial, Helvetica; font-size: 10pt; color: #FFFF00; font-weight:bold }
.yellowTitle          { font-family: Verdana, Arial, Helvetica; font-size: 14pt; color: #FFFF00; font-weight:bold; font-style:italic }
-->
</style>
</head>

<body bgcolor="#000000">

<div align="center">
    <center>
    <table border="1" width="640" cellspacing="0" cellpadding="5" bgcolor="#000080" bordercolor="#FFFFFF">
        <tr>
            <td align="center" valign="top">
                <p class="yellowTitle">Login Authentication Error</p>
                <table cellspacing="1" width="100%">
                    <tr>
                        <td align="center" valign="top">
                            <p class="navBar">The area you are attempting to access
                            requires that you login first.</p>
                            <p><a class="return" href="<?php print $rootDir; ?>bbslogin.php?url=<?php print urlencode(GetCurrentURL()); ?>">Click here to login.</a></p>
                        </td>
                    </tr>
                </table>
                <p><span class="navBar">This page is powered
                by <a class="navBar" href="http://www.vadvphp.com">VADV-PHP</a></span></p>
                <p class="infoBarBody">If you experience any problems
                with this website or need help, contact the
                <a class="infoBarBody" href="mailto:<?php print $CONFIG['webmaster']; ?>">webmaster</a>.<br />
                VADV-PHP Copyright  2002-<?php print date('Y'); ?> Aspect
                Technologies. All Rights Reserved.</p>
            </td>
        </tr>
    </table>
    </center>
</div>

</body>

</html>
<?php
}


/**
* Va-Functions.inc.php :: WMLSpecialChars()
*
* This function will take the string and convert any characters within it to
* be WML compatiable.
*
* @param string $text
* @return string
*/
function WMLSpecialChars($string) {
    if (!is_string($string)) {
        $string = (string) $string;
    }

    $string = str_replace('"', '&#34;', $string);
    $string = str_replace('&', '&#38;', $string);
    $string = str_replace("'", '&#39;', $string);
    $string = str_replace('<', '&#60;', $string);
    $string = str_replace('>', '&#62;', $string);
    $string = str_replace('  ', '&#160;&#160;', $string);

    $string = str_replace('$', '$$', $string);

    return $string;                         // Return the new string.
}


/**
* Va-Functions.inc.php :: WriteCfgFile()
*
* This function will create a new configuration file based on the file string
* passed.
*
* @param string $file
* @param string $fileString
* @return mixed
*/
function WriteCfgFile($file, $fileString) {
    if (!is_string($file)) {
        $file = (string) $file;
    }
    if (!is_string($fileString)) {
        $fileString = (string) $fileString;
    }

    $status = FALSE;

    if (version_compare(phpversion(), '5.0.0', '>=')) {
        $status = file_put_contents($file, $fileString);
    } else {
        $fp = fopen($file, 'wb');                                       // Open the file.
        if (flock($fp, LOCK_EX | LOCK_NB)) {                            // Lock the file.
            $status = fwrite($fp, $fileString, strlen($fileString));    // Write the new file.
            flock($fp, LOCK_UN);                                        // Unlock the file.
        }
        fclose($fp);                            // Close the file.
    }

    return $status;                         // Return the write status.
}


/**
* Va-Functions.inc.php :: WriteLog()
*
* This function will log a line into the specified file which is stored in
* the BBS's data directory. The logfile is a filename. MaxSize is an optional
* parameter which can be used to reset the log file if it is larger than the
* value (in bytes).
*
* @param string $logFilename
* @param string $line
* @param integer $maxSize
* @return mixed
*/
function WriteLog($logFilename, $line, $maxSize = 0) {
    if (!is_string($logFilename)) {
        $logFilename = (string)  $logFilename;
    }
    if (!is_string($line)) {
        $line        = (string)  $line;
    }
    if (!is_int($maxSize)) {
        $maxSize     = (integer) $maxSize;
    }

    global $VA_SYSPATHS;

    $logFile = $VA_SYSPATHS['data'] . '/' . $logFilename;

    // If log is larger than $maxSize then reset it.
    if (($maxSize != 0) && (is_file($logFile)) && (filesize($logFile) >= $maxSize)) {
        $oldFile = current(explode('.', basename($logFilename)));
        $oldFile = $VA_SYSPATHS['data'] . '/' . $oldFile . '.OLD';  // End the old file in .old
        if (is_file($oldFile)) {                                    // If a previous old file exists, delete it.
            unlink($oldFile);
        }
        copy($logFile, $oldFile);                                   // Copy the current log.
        unlink($logFile);                                           // Delete the current log.
    }
    
    if (!is_file($logFile)) {
        $fp = fopen($logFile, 'w');
        flock($fp, LOCK_EX);
        fwrite($fp, "\r\n");
        fwrite($fp, "VADV-PHP Event Log\r\n");
        fwrite($fp, "-------------------------------------------------------------------------------\r\n");
        fwrite($fp, "Date           Time     IP Address         Description\r\n");
        fwrite($fp, "-------------------------------------------------------------------------------\r\n");
        flock($fp, LOCK_UN);
        fclose($fp);
    }

    $fp = fopen($logFile, 'a');
    flock($fp, LOCK_EX);                    // Lock the file.

    $ip = '[' . trim($_SERVER['REMOTE_ADDR']) . ']';
    $ip = str_pad($ip, 17, ' ');

    if (IsHttps()) {
        $line = $line . ' (SSL)';
    }

    $date   = date('D') . ' ' . date('Y') . '-' . date('m') . '-' . date('d');
    $time   = date('H') . ':' . date('i') . ':' . date('s');
    $line   = $date . ' ' . $time . ' ' . $ip . '  ' . $line . "\r\n";
    $status = fwrite($fp, $line);           // Write the line to the file.

    flock($fp, LOCK_UN);                    // Unlock the file.
    fclose($fp);                            // Close the file.
    
    return $status;                         // Return the result.
}

?>