<?php
/*
 * ----------------------------------------------------------------------------
 * Va-Syspaths.inc.php                                               07/16/2006
 * This script will read the syspaths.cfg file and place its information into
 * an array for global use.
 * ----------------------------------------------------------------------------
 * Copyright (C) 2002-2005 Steve Winn. All Rights Reserved.
 *
 * The source code contains a fair amount of comments, which may aid anyone
 * attempting to follow the program or to modify it.
 *
 * For licensing and copyright information, view LICENSE.TXT.
 * ----------------------------------------------------------------------------
 */

require_once('include_dir.php');                    // Get include directory.
require_once($includeDir . 'va-functions.inc.php'); // A VADV-PHP script.


/**
* Va-Syspaths.inc.php :: ConvertDir()
*
* This function will convert a local directory into a network address.
*
* @param string $hostname
* @param string $dir
* @return string
*/
function ConvertDir($hostname, $dir) {
    if (!is_string($hostname)) {
        $hostname = (string) $hostname;
    }
    if (!is_string($dir)) {
        $dir      = (string) $dir;
    }

    $dir   = str_replace("\\", '/', $dir);  // Change to forwardslashes.
    $drive = substr($dir, 0, 1);            // Get the drive letter.
    $line  = substr($dir, 2);               // Get the rest of the path.
    $path  = '//' . trim($hostname) . '/' . trim($drive) . trim($line);
    $path  = str_replace('/', "\\", $path); // Change to backslashes
    
    return $path;                           // Return the new path.
}


/**
* Va-Syspaths.inc.php :: ConvertUnc()
*
* This function will convert a UNC path into a local directory. For Windows
* systems only.
*
* @param string $unc
* @return string
*/
function ConvertUnc($unc) {
    if (!is_string($unc)) {
        $unc = (string) $unc;
    }

    $unc   = str_replace("\\", '/', $unc);  // Change to forwardslashes.
    $pos   = strpos($unc, '/', 2);          // Get the position of the drive letter.
    $drive = substr($unc, $pos + 1, 1);     // Get the drive letter.
    $dir   = substr($unc, $pos + 3);        // Get the rest of the path.
    if ($dir == '') {
        $path  = $drive . ':';
    } else {
        $path  = $drive . ':/' . $dir;
    }
    $path  = str_replace('/', "\\", $path); // Change to backslashes.
    
    return $path;                           // Return the new path.
}


/**
* Va-Syspaths.inc.php :: GetConfig()
*
* This function will read the VADV-PHP configuration file (vadv-php_config.php)
* and place its variables into an array.
*
* @return array
*/
function GetConfig() {
    // Set the default values.
    $config = array();
    $config['banned']        = '';              // Banned user accounts.
    $config['bbsdir']        = 'c:/va';         // BBS directory.
    $config['cookielife']    = 7;               // Cookie life span.
    $config['filehost']      = '';              // Network file directory.
    $config['fileupload']    = 0;               // File uploads.
    $config['fileuploadsl']  = 255;             // File upload security level.
    $config['guestacct']     = 0;               // Guest account number.
    $config['hostname']      = '';              // Network BBS directory.
    $config['newslist']      = TRUE;            // NewsList support.
    $config['newuserreq']    = 141;             // Required new user information.
    $config['postitnt']      = TRUE;            // Post-It! Network support.
    $config['purgetime']     = 0;               // Purge time.
    $config['purgeminlevel'] = 0;               // Purge minimum security level.
    $config['purgemaxlevel'] = 0;               // Purge maximum security level.
    $config['refresh']       = 0;               // Refresh time.
    $config['sslredirect']   = '';              // URL for SSL site.
    $config['sysnews']       = TRUE;            // System News support.
    $config['syspass']       = TRUE;            // System password for Vconfig.
    $config['telnet']        = '';              // Telnet address.
    $config['useraccflags']  = '';              // Guest access flags.
    $config['userage']       = 0;               // Guest age.
    $config['usersl']        = 15;              // Guest security level.
    $config['webmaster']     = 'webmaster@';    // Webmaster email address.

    global $includeDir;
    include($includeDir . 'vadv-php_config.php');   // Include the configuration file.
    
    // Tidy it up a bit...
    $config['banned']        = trim($config['banned']);
    $config['bbsdir']        = trim($config['bbsdir']);
    $config['cookielife']    = (integer) $config['cookielife'];
    $config['filehost']      = trim($config['filehost']);
    $config['fileupload']    = (boolean) $config['fileupload'];
    $config['fileuploadsl']  = (integer) $config['fileuploadsl'];
    $config['guestacct']     = (integer) $config['guestacct'];
    $config['hostname']      = trim($config['hostname']);
    $config['newslist']      = (boolean) $config['newslist'];
    $config['newuserreq']    = (integer) $config['newuserreq'];
    $config['postitnt']      = (boolean) $config['postitnt'];
    $config['purgetime']     = (integer) $config['purgetime'];
    $config['purgeminlevel'] = (integer) $config['purgeminlevel'];
    $config['purgemaxlevel'] = (integer) $config['purgemaxlevel'];
    $config['refresh']       = (integer) $config['refresh'];
    $config['sslredirect']   = trim($config['sslredirect']);
    $config['sysnews']       = (boolean) $config['sysnews'];
    $config['syspass']       = (boolean) $config['syspass'];
    $config['telnet']        = trim($config['telnet']);
    $config['useraccflags']  = trim(strtoupper(str_replace(' ', '', $config['useraccflags'])));
    $config['userage']       = (integer) $config['userage'];
    $config['usersl']        = (integer) $config['usersl'];
    $config['webmaster']     = trim($config['webmaster']);

    // If the guest account is used, then ban that account.
    if ($config['guestacct'] > 0) {
        $config['banned'] = $config['guestacct'] . ',' . $config['banned'];
    }

    // If the purge max security level is zero, then set it to 254.
    if ($config['purgemaxlevel'] == 0) {
        $config['purgemaxlevel'] = 254;
    }

    return $config;
}


/**
* Va-Syspaths.inc.php :: GetSysPaths()
*
* This function will retrieve the vadv system paths from the syspaths.cfg file
* and return the paths in an array.
*
* @return array
*/
function GetSysPaths() {
    global $CONFIG;                         // Include the global variable.

    $fileString = ReadSysPaths();           // Read the file.

    $syspaths = array();
    $syspaths['main'] = GetString($fileString, 1, 40);
    if ($CONFIG['hostname'] != '') $syspaths['main']    = ConvertDir($CONFIG['hostname'], $syspaths['main']);
    $syspaths['script'] = GetString($fileString, 41, 40);
    if ($CONFIG['hostname'] != '') $syspaths['script']  = ConvertDir($CONFIG['hostname'], $syspaths['script']);
    $syspaths['data'] = GetString($fileString, 81, 40);
    if ($CONFIG['hostname'] != '') $syspaths['data']    = ConvertDir($CONFIG['hostname'], $syspaths['data']);
    $syspaths['network'] = GetString($fileString, 121, 40);
    if ($CONFIG['hostname'] != '') $syspaths['network'] = ConvertDir($CONFIG['hostname'], $syspaths['network']);
    $syspaths['text'] = GetString($fileString, 161, 40);
    if ($CONFIG['hostname'] != '') $syspaths['text']    = ConvertDir($CONFIG['hostname'], $syspaths['text']);
    $syspaths['control'] = GetString($fileString, 201, 40);
    if ($CONFIG['hostname'] != '') $syspaths['control'] = ConvertDir($CONFIG['hostname'], $syspaths['control']);
    $syspaths['temp'] = GetString($fileString, 241, 40);
    if ($CONFIG['hostname'] != '') $syspaths['temp']    = ConvertDir($CONFIG['hostname'], $syspaths['temp']);
    $syspaths['db'] = GetString($fileString, 281, 40);
    if ($CONFIG['hostname'] != '') $syspaths['db']      = ConvertDir($CONFIG['hostname'], $syspaths['db']);
    $syspaths['sysop'] = GetString($fileString, 321, 40);
    if ($CONFIG['hostname'] != '') $syspaths['sysop']   = ConvertDir($CONFIG['hostname'], $syspaths['sysop']);
    $syspaths['fdx'] = GetString($fileString, 361, 40);
    if ($CONFIG['hostname'] != '') $syspaths['fdx']     = ConvertDir($CONFIG['hostname'], $syspaths['fdx']);
    
    return $syspaths;                       // Return the BBS system paths.
}


/**
* Va-SysPaths.inc.php :: PutSysPaths()
*
* This function will write updated information to the control.dat file, which
* is used to determine the current users online.
*
* @param string $fileString
* @param array $configVariable
* @return string
*/
function PutSysPaths($fileString, $configVariable) {
    if (!is_string($fileString)) {
        $fileString = (string) $fileString;
    }
    if (!is_array($configVariable)) {
        trigger_error(VA_Error(1));
    }

    if (isset($configVariable['main']))    $fileString = PutString($fileString, $configVariable['main'], 1, 40);
    if (isset($configVariable['script']))  $fileString = PutString($fileString, $configVariable['script'], 41, 40);
    if (isset($configVariable['data']))    $fileString = PutString($fileString, $configVariable['data'], 81, 40);
    if (isset($configVariable['network'])) $fileString = PutString($fileString, $configVariable['network'], 121, 32);
    if (isset($configVariable['text']))    $fileString = PutString($fileString, $configVariable['text'], 161, 32);
    if (isset($configVariable['control'])) $fileString = PutString($fileString, $configVariable['control'], 201, 3);
    if (isset($configVariable['temp']))    $fileString = PutString($fileString, $configVariable['temp'], 241, 10);
    if (isset($configVariable['db']))      $fileString = PutString($fileString, $configVariable['db'], 281, 12);
    if (isset($configVariable['sysop']))   $fileString = PutString($fileString, $configVariable['sysop'], 321, 12);
    if (isset($configVariable['fdx']))     $fileString = PutString($fileString, $configVariable['fdx'], 361, 12);

    return $fileString;
}


/**
* Va-SysPaths.inc.php :: ReadSysPaths()
*
* This function will read the syspaths.cfg file.
*
* @return string
*/
function ReadSysPaths() {
    global $CONFIG;                         // Include the global variable.

    // Check to see if a hostname was configured for a network path.
    if ($CONFIG['hostname'] != '') {
        $file = ConvertDir($CONFIG['hostname'], $CONFIG['bbsdir']) . '/SYSPATHS.CFG';
    } else {
        $file = $CONFIG['bbsdir'] . '/SYSPATHS.CFG';
    }

    if (!is_file($file)) {
        trigger_error(VA_Error(0));
    }
    $fileString = ReadCfgFile($file);       // Read the file.

    return $fileString;                     // Return the file string.
}


/**
* Va-SysPaths.inc.php :: WriteConfig()
*
* This function will write a new vadv-php_config.php file based on the values in
* the configuration variable.
*
* @param array $configVariable
* @return mixed
*/
function WriteConfig($configVariable) {
    if (!is_array($configVariable)) {
        trigger_error(VA_Error(1));
    }

    global $includeDir;
    
    $filename = $includeDir . 'vadv-php_config.php';
    if (!is_file($filename)) {
        return FALSE;
    }

    $file     = file($filename);            // Read the file into an array.

    $newFile = '';
    for ($x = 0; $x < count($file) - 1; $x++) {
        $line = $file[$x];
        $comment = strpos($line, '//');
        if ($comment !== FALSE) {
            $line = trim(substr($line, 0, $comment - 1));
        }
        $line = trim(strtolower($line));
        if (($line != '') || ($line != '?'. '>')) {
            $a = strpos($line, '$config[');
            if ($a !== FALSE) {
                $line = trim(substr($line, $a + 9));
                $a = strpos($line, ']');
                $line = trim(substr($line, 0, $a - 1));
                reset($configVariable);
                while (list($key, $value) = each($configVariable)) {
                    if (strcmp(trim(strtolower($key)), trim($line)) == 0) {
                        $file[$x] = '$' . "config['" . $key . "'] = '" . trim($value) . "'" . ';' . "\r\n";
                        unset($configVariable[$key]);
                    }
                }
            }
        }
        $newFile = $newFile . $file[$x];
    }

    // Any left over config settings, append to the end of the file.
    reset($configVariable);
    while (list($key, $value) = each($configVariable)) {
        $newFile = $newFile . '$' . "config['" . $key . "'] = '" . trim($value) . "'" . ';' . "\r\n";
    }
    
    $newFile = $newFile . '?' . '>';

    // Write the new config file.
    $status = WriteCfgFile($filename, $newFile);
    
    return $status;                         // Return the status.
}


/**
* Va-SysPaths.inc.php :: WriteSysPaths()
*
* This function will write a new syspaths.cfg file based on the file string
* passed to it. If the file string is not in the correct format, then your
* syspaths.cfg file will become corrupt and the BBS could stop functioning!
*
* @param string $fileString
* @return mixed
*/
function WriteSysPaths($fileString) {
    if (!is_string($fileString)) {
        $fileString = (string) $fileString;
    }

    global $VA_SYSPATHS;                    // Include the global variable.

    // Set the filename.
    $file = $VA_SYSPATHS['main'] . '/SYSPATHS.CFG';
    $status = WriteCfgFile($file, $fileString);

    return $status;                         // Return the result status.
}


// Read the configuration file.
$CONFIG = GetConfig();                      // Get the configuration using vadv-php_config.php.

// Load all of the syspaths.cfg variables automatically.
$VA_SYSPATHS = GetSysPaths();

?>